<?php
declare(strict_types=1);

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

require_login();
require_role('admin');

function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function detect_password_column(PDO $pdo): string {
  try {
    $cols = $pdo->query("DESCRIBE users")->fetchAll(PDO::FETCH_ASSOC);
    $names = array_map(fn($r) => (string)($r['Field'] ?? ''), $cols);
    if (in_array('password_hash', $names, true)) return 'password_hash';
    if (in_array('password', $names, true)) return 'password';
  } catch (Throwable $e) {}
  return 'password_hash';
}

function users_has_history(PDO $pdo): bool {
  try {
    $cols = $pdo->query("DESCRIBE users")->fetchAll(PDO::FETCH_ASSOC);
    foreach ($cols as $c) {
      if ((string)($c['Field'] ?? '') === 'history') return true;
    }
  } catch (Throwable $e) {}
  return false;
}

$pwdcol = detect_password_column($pdo);
$hasHistory = users_has_history($pdo);

$msg = '';
$err = '';

/* -------------------- Actions -------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = (string)($_POST['action'] ?? '');

  try {
    if ($action === 'add') {
      $name = trim((string)($_POST['name'] ?? ''));
      $initials = normalize_initials((string)($_POST['initials'] ?? ''));
      $email = normalize_email((string)($_POST['email'] ?? ''));
      $role = (string)($_POST['role'] ?? 'user');

      if ($name === '' || $initials === '' || $email === '' || $role === '') {
        throw new RuntimeException('All fields are required.');
      }
      if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new RuntimeException('Invalid email.');
      }
      if (!initials_valid($initials)) {
        throw new RuntimeException('Initials must be 2–3 uppercase letters.');
      }
      if (!in_array($role, ['admin','manager','user'], true)) {
        throw new RuntimeException('Invalid role.');
      }

      // Unique email
      $st = $pdo->prepare("SELECT COUNT(*) FROM users WHERE email = :e");
      $st->execute([':e' => $email]);
      if ((int)$st->fetchColumn() > 0) {
        throw new RuntimeException('Email already exists.');
      }

      // Unique initials
      $st = $pdo->prepare("SELECT COUNT(*) FROM users WHERE initials = :i");
      $st->execute([':i' => $initials]);
      if ((int)$st->fetchColumn() > 0) {
        throw new RuntimeException('Initials already exist.');
      }

      // Password remains NULL/empty until first login sets it
      $sql = "INSERT INTO users (name, initials, email, role, `$pwdcol`" . ($hasHistory ? ", history" : "") . ")
              VALUES (:n, :i, :e, :r, NULL" . ($hasHistory ? ", NULL" : "") . ")";
      $st = $pdo->prepare($sql);
      $st->execute([':n'=>$name, ':i'=>$initials, ':e'=>$email, ':r'=>$role]);

      $msg = 'User created. They must set a password on first login.';
    }

    if ($action === 'role') {
      $id = (int)($_POST['id'] ?? 0);
      $role = (string)($_POST['role'] ?? '');
      if ($id <= 0) throw new RuntimeException('Invalid user.');
      if (!in_array($role, ['admin','manager','user'], true)) throw new RuntimeException('Invalid role.');

      $st = $pdo->prepare("UPDATE users SET role = :r WHERE id = :id");
      $st->execute([':r'=>$role, ':id'=>$id]);

      $msg = 'Role updated.';
    }

    if ($action === 'reset_password') {
      $id = (int)($_POST['id'] ?? 0);
      if ($id <= 0) throw new RuntimeException('Invalid user.');

      // Clear password so next login forces set-password
      $st = $pdo->prepare("UPDATE users SET `$pwdcol` = NULL WHERE id = :id");
      $st->execute([':id'=>$id]);

      $msg = 'Password cleared. User will be prompted to set a new password on next login.';
    }

    if ($action === 'clear_history') {
      $id = (int)($_POST['id'] ?? 0);
      if ($id <= 0) throw new RuntimeException('Invalid user.');
      if (!$hasHistory) throw new RuntimeException('History column not available.');

      $st = $pdo->prepare("UPDATE users SET history = NULL WHERE id = :id");
      $st->execute([':id'=>$id]);

      $msg = 'History cleared.';
    }

  } catch (Throwable $e) {
    $err = $e->getMessage();
  }
}

/* -------------------- Load users -------------------- */
$cols = "id, name, initials, email, role, `$pwdcol` AS _pwd";
if ($hasHistory) $cols .= ", history";

$users = $pdo->query("SELECT $cols FROM users ORDER BY role DESC, initials ASC, name ASC")->fetchAll(PDO::FETCH_ASSOC);

include __DIR__ . '/header.php';
?>
<section>
  <div class="toolbar" style="justify-content:space-between;align-items:center">
    <h2 style="margin:0">Users</h2>
  </div>

  <?php if ($msg): ?>
    <p class="note"><?= h($msg) ?></p>
  <?php endif; ?>
  <?php if ($err): ?>
    <p class="note" style="border:1px solid rgba(255,0,0,.25);padding:10px;border-radius:12px;">
      <?= h($err) ?>
    </p>
  <?php endif; ?>

  <div class="card" style="margin-bottom:12px;">
    <h3 style="margin:0 0 10px 0;">Add user</h3>
    <form method="post" style="display:flex;gap:10px;flex-wrap:wrap;align-items:flex-end;">
      <input type="hidden" name="action" value="add">
      <div style="min-width:220px;flex:1;">
        <label>Name</label>
        <input class="search-input" name="name" required>
      </div>
      <div style="width:120px;flex:1;">
        <label>Initials</label>
        <input class="search-input" name="initials" maxlength="3" style="text-transform:uppercase" required>
      </div>
      <div style="min-width:260px;flex:1;">
        <label>Email</label>
        <input class="search-input" name="email" type="email" required>
      </div>
      <div style="width:160px;">
        <label>Role</label>
        <select class="search-input" name="role" required>
          <option value="user">user</option>
          <option value="manager">manager</option>
          <option value="admin">admin</option>
        </select>
      </div>
      <div>
        <button class="btn primary" type="submit">add</button>
      </div>
    </form>
    <p class="note" style="margin:10px 0 0 0;">
      Users are created without a password. On first login, they will be asked to set one.
      Use “reset password” to force a new password.
    </p>
  </div>

  <div class="table-wrap">
    <table>
      <thead>
        <tr>
          <th>actions</th>
          <th>id</th>
          <th>name</th>
          <th>initials</th>
          <th>email</th>
          <th>role</th>
          <th>has password</th>
          <?php if ($hasHistory): ?>
            <th>history</th>
          <?php endif; ?>
        </tr>
      </thead>
      <tbody>
        <?php foreach ($users as $u): ?>
          <?php
            $uid = (int)($u['id'] ?? 0);
            $hasPwd = !empty($u['_pwd']);
          ?>
          <tr>
            <td class="actions-cell">
              <form method="post" style="display:inline-flex;gap:6px;flex-wrap:wrap;">
                <input type="hidden" name="id" value="<?= $uid ?>">
                <input type="hidden" name="action" value="role">
                <select class="search-input" name="role" onchange="this.form.submit()">
                  <option value="user" <?= (($u['role'] ?? '')==='user')?'selected':''; ?>>user</option>
                  <option value="manager" <?= (($u['role'] ?? '')==='manager')?'selected':''; ?>>manager</option>
                  <option value="admin" <?= (($u['role'] ?? '')==='admin')?'selected':''; ?>>admin</option>
                </select>
              </form>

              <form method="post" style="display:inline;">
                <input type="hidden" name="id" value="<?= $uid ?>">
                <input type="hidden" name="action" value="reset_password">
                <button class="btn small" type="submit">reset password</button>
              </form>

              <?php if ($hasHistory): ?>
                <form method="post" style="display:inline;">
                  <input type="hidden" name="id" value="<?= $uid ?>">
                  <input type="hidden" name="action" value="clear_history">
                  <button class="btn small ghost" type="submit" title="clear login history">clear history</button>
                </form>
              <?php endif; ?>
            </td>

            <td><?= $uid ?></td>
            <td><?= h($u['name'] ?? '') ?></td>
            <td><?= h($u['initials'] ?? '') ?></td>
            <td><?= h($u['email'] ?? '') ?></td>
            <td><?= h($u['role'] ?? '') ?></td>
            <td><?= $hasPwd ? 'yes' : 'no' ?></td>

            <?php if ($hasHistory): ?>
              <td style="white-space:pre-wrap;min-width:260px;max-width:520px;">
                <?php
                  $hist = (string)($u['history'] ?? '');
                  if ($hist === '') {
                    echo '<span class="note">(empty)</span>';
                  } else {
                    // Display as-is (CR/LF preserved)
                    echo h($hist);
                  }
                ?>
              </td>
            <?php endif; ?>
          </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
  </div>
</section>

<?php include __DIR__ . '/footer.php'; ?>
