<?php
declare(strict_types=1);

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

require_login();
$me = current_user();

/*
  ==========================================================
  API check — ZeroBounce "chi" button
  ==========================================================

  Visibility mode for the "chi" button:
    - ACTIVE   : visible to all users
    - INACTIVE : not visible
    - ADMIN    : visible only to admin users

  Default requested: ACTIVE.
*/
$CHI_BUTTON_MODE = 'ACTIVE'; // ACTIVE | INACTIVE | ADMIN

/*
  ZeroBounce configuration (keep server-side)
*/
$ZERObounce_API_URL = 'https://api.zerobounce.net/v2/validate';
$ZERObounce_API_KEY = '1c860d8460fe4c068c0d5be7f901a14a';

/* ---------- helpers ---------- */
if (!function_exists('json_out')) {
  function json_out($data, int $code = 200): void {
    if (!headers_sent()) {
      http_response_code($code);
      header('Content-Type: application/json; charset=utf-8');
      header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
    }
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
  }
}
if (!function_exists('h')) {
  function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}

/*
  IMPORTANT:
  - Do NOT redeclare sanitize_sort() because includes/db.php may already define it.
  - We will call sanitize_sort() from db.php if present; otherwise, fallback locally.
*/
if (!function_exists('sanitize_sort')) {
  function sanitize_sort(string $field, array $allowed): string {
    return in_array($field, $allowed, true) ? $field : 'id';
  }
}

/* ---------- ZeroBounce helpers (namespaced to avoid collisions) ---------- */
if (!function_exists('vm_zb_call_validate')) {
  function vm_zb_call_validate(string $baseUrl, string $apiKey, string $email, string $ipAddress = ''): array {
    $email = trim($email);
    $url = $baseUrl
      . '?api_key=' . rawurlencode($apiKey)
      . '&email=' . rawurlencode($email)
      . '&ip_address=' . rawurlencode($ipAddress);

    $body = '';
    $httpCode = 0;

    if (function_exists('curl_init')) {
      $ch = curl_init();
      curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 20,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
      ]);
      $body = (string)curl_exec($ch);
      $httpCode = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
      curl_close($ch);
    } else {
      $ctx = stream_context_create([
        'http' => [
          'method' => 'GET',
          'timeout' => 20,
          'header' => "Accept: application/json\r\n",
        ]
      ]);
      $body = (string)@file_get_contents($url, false, $ctx);
    }

    $json = null;
    if ($body !== '') {
      $json = json_decode($body, true);
    }

    if (!is_array($json)) {
      return [
        'ok' => false,
        'http_code' => $httpCode,
        'error' => 'invalid json',
        'raw' => $body,
      ];
    }

    if (isset($json['error']) && is_string($json['error']) && $json['error'] !== '') {
      return [
        'ok' => false,
        'http_code' => $httpCode,
        'error' => $json['error'],
      ];
    }

    return [
      'ok' => true,
      'http_code' => $httpCode,
      'data' => $json,
    ];
  }
}

if (!function_exists('vm_comment_line_validated')) {
  function vm_comment_line_validated(array $zbJson): string {
    $ts = date('Y-m-d H:i');

    $first = trim((string)($zbJson['firstname'] ?? ''));
    $last  = trim((string)($zbJson['lastname'] ?? ''));
    $gender = trim((string)($zbJson['gender'] ?? ''));
    $country = trim((string)($zbJson['country'] ?? ''));

    // Content only (no labels)
    $parts = [];
    if ($first !== '') $parts[] = $first;
    if ($last !== '')  $parts[] = $last;
    if ($gender !== '') $parts[] = $gender;
    if ($country !== '') $parts[] = $country;

    $tail = implode(' ', $parts);
    return trim($ts . ' validated' . ($tail !== '' ? ' ' . $tail : ''));
  }
}

if (!function_exists('vm_comment_line_invalid_email')) {
  function vm_comment_line_invalid_email(): string {
    $ts = date('Y-m-d H:i');
    return $ts . ' invalid email';
  }
}

$role = (string)($me['role'] ?? 'user');
$is_admin   = ($role === 'admin');
$is_manager = ($role === 'manager');

/* ---------- Column visibility by role ----------
   - user: hide initials, private
   - manager: show initials, hide private
   - admin: show all
*/
$show_initials = ($is_admin || $is_manager);
$show_private  = $is_admin;

/*
  Canonical field order + job->role change:
  role must appear immediately after status
*/
$ordered_fields = [
  'company_name','name','mf','email','wrong','linkedIn','status','role','language','country','hq_city',
  'comment','history','sector','grupo','website','general_email','last_date','last_template','initials','private'
];

/* ---------- AJAX endpoints ---------- */
if (!empty($_GET['action'])) {
  try {
    $action = (string)$_GET['action'];

    // visibility clause
    $vis = companies_visibility_clause($me);
    $visClause = $vis[0];
    $visParams = $vis[1];

    // Soft-delete: never show deleted rows on list.php endpoints
    $notDeletedSql = "IFNULL(`del`,0)=0";

    if ($action === 'countries') {
      $st = $pdo->prepare("SELECT DISTINCT IFNULL(country,'') AS v FROM companies WHERE ($visClause) AND $notDeletedSql ORDER BY v ASC");
      $st->execute($visParams);
      $rows = $st->fetchAll(PDO::FETCH_COLUMN);
      $out = [];
      foreach ($rows as $v) {
        $v = (string)$v;
        if (trim($v) === '') continue;
        $out[] = $v;
      }
      json_out(['countries' => $out]);
    }

    if ($action === 'grupos') {
      $st = $pdo->prepare("SELECT DISTINCT IFNULL(grupo,'') AS v FROM companies WHERE ($visClause) AND $notDeletedSql ORDER BY v ASC");
      $st->execute($visParams);
      $rows = $st->fetchAll(PDO::FETCH_COLUMN);
      $out = [];
      foreach ($rows as $v) {
        $v = (string)$v;
        if (trim($v) === '') continue;
        $out[] = $v;
      }
      json_out(['grupos' => $out]);
    }

    if ($action === 'sectors') {
      $st = $pdo->prepare("SELECT DISTINCT IFNULL(sector,'') AS v FROM companies WHERE ($visClause) AND $notDeletedSql ORDER BY v ASC");
      $st->execute($visParams);
      $rows = $st->fetchAll(PDO::FETCH_COLUMN);
      $out = [];
      foreach ($rows as $v) {
        $v = (string)$v;
        if (trim($v) === '') continue;
        $out[] = $v;
      }
      json_out(['sectors' => $out]);
    }

    if ($action === 'list') {
      $country = (string)($_GET['country'] ?? 'all');
      $status  = (string)($_GET['status'] ?? 'all');
      $grupo   = (string)($_GET['grupo'] ?? 'all');
      $sector  = (string)($_GET['sector'] ?? 'all');
      $q       = trim((string)($_GET['q'] ?? ''));

      $sortable = array_merge(['id'], $ordered_fields);
      $sort = sanitize_sort((string)($_GET['sort'] ?? 'id'), $sortable);
      $dir  = (isset($_GET['dir']) && strtoupper((string)$_GET['dir']) === 'DESC') ? 'DESC' : 'ASC';

      $where = [ $visClause, $notDeletedSql ];
      $params = $visParams;

      if ($country !== '' && $country !== 'all') {
        $where[] = "country = :c";
        $params[':c'] = $country;
      }
      if ($status !== '' && $status !== 'all') {
        $where[] = "status = :s";
        $params[':s'] = $status;
      }
      if ($grupo !== '' && $grupo !== 'all') {
        $where[] = "grupo = :g";
        $params[':g'] = $grupo;
      }
      if ($sector !== '' && $sector !== 'all') {
        $where[] = "sector = :sec";
        $params[':sec'] = $sector;
      }

      if ($q !== '') {
        $params[':q'] = "%$q%";
        $where[] = "("
          . "CAST(id AS CHAR) LIKE :q OR company_name LIKE :q OR name LIKE :q OR mf LIKE :q OR email LIKE :q OR "
          . "linkedIn LIKE :q OR status LIKE :q OR role LIKE :q OR language LIKE :q OR country LIKE :q OR hq_city LIKE :q OR "
          . "comment LIKE :q OR history LIKE :q OR sector LIKE :q OR grupo LIKE :q OR website LIKE :q OR "
          . "general_email LIKE :q OR last_template LIKE :q OR initials LIKE :q OR "
          . "DATE_FORMAT(last_date, '%Y-%m-%d %H:%i:%s') LIKE :q"
          . ")";
      }

      $sql = "SELECT * FROM companies WHERE " . implode(" AND ", $where) . " ORDER BY `$sort` $dir";
      $stmt = $pdo->prepare($sql);
      $stmt->execute($params);
      $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

      // Total visible (unfiltered) for the user, excluding deleted
      $st2 = $pdo->prepare("SELECT COUNT(*) FROM companies WHERE ($visClause) AND $notDeletedSql");
      $st2->execute($visParams);
      $totalVisible = (int)$st2->fetchColumn();

      json_out([
        'ok' => true,
        'rows' => $rows,
        'filtered' => count($rows),
        'total' => $totalVisible
      ]);
    }

    if ($action === 'get' && isset($_GET['id'])) {
      $id = (int)$_GET['id'];
      $st = $pdo->prepare("SELECT * FROM companies WHERE id=:id LIMIT 1");
      $st->execute([':id'=>$id]);
      $row = $st->fetch(PDO::FETCH_ASSOC);
      if (!$row) json_out(['ok'=>false,'error'=>'not found'], 404);
      if (!can_access_company_row($row, $me)) json_out(['ok'=>false,'error'=>'forbidden'], 403);
      json_out(['ok'=>true,'row'=>$row]);
    }

    if ($action === 'update' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $id = (int)($_POST['id'] ?? 0);
      $field = (string)($_POST['field'] ?? '');
      $value = $_POST['value'] ?? null;

      if ($id <= 0) json_out(['ok'=>false,'error'=>'invalid id'], 400);

      $st = $pdo->prepare("SELECT * FROM companies WHERE id=:id LIMIT 1");
      $st->execute([':id'=>$id]);
      $row = $st->fetch(PDO::FETCH_ASSOC);
      if (!$row) json_out(['ok'=>false,'error'=>'not found'], 404);
      if (!can_access_company_row($row, $me)) json_out(['ok'=>false,'error'=>'forbidden'], 403);

      $allowed = [
        'company_name','name','mf','email','wrong','linkedIn','status','role','language','country','hq_city',
        'comment','history','sector','grupo','website','general_email','last_date','last_template'
      ];
      if ($is_admin) {
        $allowed[] = 'initials';
        $allowed[] = 'private';
      }

      if (!in_array($field, $allowed, true)) json_out(['ok'=>false,'error'=>'invalid field'], 400);

      if ($field === 'wrong' || $field === 'private') {
        $value = ((string)$value === '1') ? '1' : '0';
      }

      if ($field === 'initials') {
        if (!$is_admin) json_out(['ok'=>false,'error'=>'forbidden'], 403);
        $value = strtoupper(trim((string)$value));
      }
      if ($field === 'private' && !$is_admin) {
        json_out(['ok'=>false,'error'=>'forbidden'], 403);
      }

      $sql = "UPDATE companies SET `$field` = :v WHERE id = :id";
      $pdo->prepare($sql)->execute([':v'=>$value, ':id'=>$id]);

      json_out(['ok'=>true]);
    }

    /*
      ZeroBounce validate endpoint
      POST: id
      - UI disables when email empty OR wrong=1 (server also enforces)
      - valid: append "<ts> validated <firstname> <lastname> <gender> <country>"
      - not valid: set wrong=1 AND append "<ts> invalid email"
    */
    if ($action === 'chi_validate' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $mode = strtoupper(trim((string)$GLOBALS['CHI_BUTTON_MODE']));
      if ($mode === 'INACTIVE') json_out(['ok'=>false,'error'=>'disabled'], 403);
      if ($mode === 'ADMIN' && !$is_admin) json_out(['ok'=>false,'error'=>'forbidden'], 403);

      $id = (int)($_POST['id'] ?? 0);
      if ($id <= 0) json_out(['ok'=>false,'error'=>'invalid id'], 400);

      $st = $pdo->prepare("SELECT * FROM companies WHERE id=:id LIMIT 1");
      $st->execute([':id'=>$id]);
      $row = $st->fetch(PDO::FETCH_ASSOC);
      if (!$row) json_out(['ok'=>false,'error'=>'not found'], 404);
      if (!can_access_company_row($row, $me)) json_out(['ok'=>false,'error'=>'forbidden'], 403);

      $email = trim((string)($row['email'] ?? ''));
      if ($email === '') json_out(['ok'=>false,'error'=>'missing email'], 400);

      if (!empty($row['wrong'])) {
        json_out(['ok'=>false,'error'=>'email marked wrong'], 400);
      }

      $apiUrl = (string)$GLOBALS['ZERObounce_API_URL'];
      $apiKey = (string)$GLOBALS['ZERObounce_API_KEY'];

      $resp = vm_zb_call_validate($apiUrl, $apiKey, $email, '');
      if (empty($resp['ok'])) {
        // Do not mutate DB on provider error
        $detail = (string)($resp['error'] ?? 'provider');
        json_out(['ok'=>false,'error'=>'zerobounce', 'detail'=>$detail], 502);
      }

      $data = (array)($resp['data'] ?? []);
      $status = strtolower(trim((string)($data['status'] ?? '')));

      if ($status === 'valid') {
        $line = vm_comment_line_validated($data);

        $pdo->prepare(
          "UPDATE companies
             SET comment = CASE WHEN comment IS NULL OR comment = '' THEN :line ELSE CONCAT(comment, '\n', :line) END
           WHERE id = :id"
        )->execute([':line' => $line, ':id' => $id]);

        $st2 = $pdo->prepare("SELECT comment, wrong FROM companies WHERE id=:id LIMIT 1");
        $st2->execute([':id' => $id]);
        $after = $st2->fetch(PDO::FETCH_ASSOC) ?: ['comment'=>'', 'wrong'=>0];

        json_out([
          'ok' => true,
          'status' => 'valid',
          'zb' => [
            'firstname' => (string)($data['firstname'] ?? ''),
            'lastname'  => (string)($data['lastname'] ?? ''),
            'gender'    => (string)($data['gender'] ?? ''),
            'country'   => (string)($data['country'] ?? ''),
          ],
          'comment' => (string)($after['comment'] ?? ''),
          'wrong' => (string)($after['wrong'] ?? '0'),
        ]);
      }

      // Any non-valid status: wrong=1 + append comment "invalid email"
      $invalidLine = vm_comment_line_invalid_email();

      $pdo->prepare(
        "UPDATE companies
            SET wrong=1,
                comment = CASE WHEN comment IS NULL OR comment = '' THEN :line ELSE CONCAT(comment, '\n', :line) END
          WHERE id = :id"
      )->execute([':line' => $invalidLine, ':id' => $id]);

      $st2 = $pdo->prepare("SELECT comment, wrong FROM companies WHERE id=:id LIMIT 1");
      $st2->execute([':id' => $id]);
      $after = $st2->fetch(PDO::FETCH_ASSOC) ?: ['comment'=>'', 'wrong'=>1];

      json_out([
        'ok' => true,
        'status' => $status !== '' ? $status : 'unknown',
        'comment' => (string)($after['comment'] ?? ''),
        'wrong' => (string)($after['wrong'] ?? '1'),
      ]);
    }

    if ($action === 'add' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $insertable = $ordered_fields;

      // users: no initials + no private
      if (!$is_admin) {
        $tmp = [];
        foreach ($insertable as $f) {
          if ($f === 'initials') continue;
          if ($f === 'private') continue;
          $tmp[] = $f;
        }
        $insertable = $tmp;
      }

      // managers: no private
      if ($is_manager) {
        $tmp = [];
        foreach ($insertable as $f) {
          if ($f === 'private') continue;
          $tmp[] = $f;
        }
        $insertable = $tmp;
      }

      $data = [];
      foreach ($insertable as $f) {
        if (array_key_exists($f, $_POST)) $data[$f] = $_POST[$f];
      }

      $data['wrong'] = (isset($data['wrong']) && ((string)$data['wrong'] === '1')) ? '1' : '0';

      if ($is_admin && array_key_exists('private', $data)) {
        $data['private'] = ((string)($data['private'] ?? '0') === '1') ? '1' : '0';
      } else {
        $data['private'] = '0';
      }

      // Soft delete default
      if (!array_key_exists('del', $data)) $data['del'] = '0';

      if ($is_admin) {
        $init = isset($data['initials']) ? strtoupper(trim((string)$data['initials'])) : '';
        $data['initials'] = ($init !== '') ? $init : strtoupper(trim((string)($me['initials'] ?? '')));
      } else {
        $data['initials'] = strtoupper(trim((string)($me['initials'] ?? '')));
      }

      if (empty($data)) json_out(['ok'=>false,'error'=>'no data'], 400);

      $cols = array_keys($data);
      $phs = [];
      foreach ($cols as $c) $phs[] = ':' . $c;

      $sql = "INSERT INTO companies (`" . implode("`,`", $cols) . "`) VALUES (" . implode(',', $phs) . ")";
      $stmt = $pdo->prepare($sql);
      $params = [];
      foreach ($cols as $c) $params[':' . $c] = $data[$c];
      $stmt->execute($params);

      json_out(['ok'=>true, 'id'=>(int)$pdo->lastInsertId()]);
    }

    // Soft delete: set companies.del = 1 (admin only)
    if ($action === 'delete' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      if (!$is_admin) json_out(['ok'=>false,'error'=>'forbidden'], 403);

      $id = (int)($_POST['id'] ?? 0);
      if ($id <= 0) json_out(['ok'=>false,'error'=>'missing id'], 400);

      $st = $pdo->prepare("SELECT * FROM companies WHERE id=:id LIMIT 1");
      $st->execute([':id'=>$id]);
      $row = $st->fetch(PDO::FETCH_ASSOC);
      if (!$row) json_out(['ok'=>false,'error'=>'not found'], 404);

      if (!can_access_company_row($row, $me)) json_out(['ok'=>false,'error'=>'forbidden'], 403);

      $pdo->prepare("UPDATE companies SET `del`=1 WHERE id=:id")->execute([':id'=>$id]);
      json_out(['ok'=>true]);
    }

    json_out(['ok'=>false,'error'=>'unknown action'], 400);

  } catch (Throwable $e) {
    json_out(['ok'=>false, 'error'=>'server', 'detail'=>$e->getMessage()], 500);
  }
}

/* ---------- Normal page render ---------- */
include __DIR__ . '/header.php';
?>
<script>
  window.VISITORMAIL_ROLE = <?= json_encode($role) ?>;
  window.VISITORMAIL_IS_ADMIN = <?= $is_admin ? 'true' : 'false' ?>;
  window.VISITORMAIL_CHI_MODE = <?= json_encode(strtoupper($CHI_BUTTON_MODE)) ?>;
</script>

<section>
  <div class="toolbar" style="display:flex;gap:10px;align-items:center;flex-wrap:wrap;">
    <select id="countryFilter" class="search-input"></select>
    <select id="statusFilter" class="search-input"></select>
    <select id="grupoFilter" class="search-input"></select>
    <select id="sectorFilter" class="search-input"></select>

    <button id="resetFilters" class="btn ghost">all</button>

    <div class="search-wrap" style="margin-left:auto;display:flex;gap:6px;align-items:center;">
      <input id="q" class="search-input" type="text" placeholder="search all fields..." autocomplete="off">
      <button id="btnClearSearch" class="btn small ghost" title="clear">×</button>
    </div>

    <button id="btnAdd" class="btn primary">add a new record</button>
  </div>

  <div class="table-wrap">
    <table id="companiesTable">
      <thead>
        <tr>
          <?php
            $cols = ['actions','id'];
            foreach ($ordered_fields as $f) {
              if ($f === 'initials' && !$show_initials) continue;
              if ($f === 'private' && !$show_private) continue;
              $cols[] = $f;
            }
            $cols[] = 'id';

            foreach ($cols as $c) {
              $label = h($c);
              $cls = ($c === 'actions') ? '' : 'sortable';
              echo "<th data-field=\"{$label}\" class=\"{$cls}\">{$label}</th>";
            }
          ?>
        </tr>
      </thead>
      <tbody></tbody>
    </table>
  </div>
</section>

<?php include __DIR__ . '/footer.php'; ?>
