<?php
// includes/auth.php — session + access control helpers

declare(strict_types=1);

require_once __DIR__ . '/db.php';

if (session_status() !== PHP_SESSION_ACTIVE) {
  session_start();
}

function normalize_email(string $email): string {
  return strtolower(trim($email));
}

function normalize_initials(string $initials): string {
  return strtoupper(trim($initials));
}

function initials_valid(string $initials): bool {
  return (bool)preg_match('/^[A-Z]{2,3}$/', $initials);
}

function is_logged_in(): bool {
  return !empty($_SESSION['visitormail_user_id']);
}

function current_user(): ?array {
  global $pdo;
  if (!is_logged_in()) return null;

  $id = (int)$_SESSION['visitormail_user_id'];
  $st = $pdo->prepare("SELECT * FROM users WHERE id = :id LIMIT 1");
  $st->execute([':id'=>$id]);
  $u = $st->fetch();
  return $u ?: null;
}

function require_login(): void {
  $path = basename(parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH));
  if ($path === 'login.php' || $path === 'logout.php') return;

  if (!is_logged_in()) {
    header('Location: login.php?next=' . urlencode($path));
    exit;
  }
}

function require_role(string $role): void {
  $u = current_user();
  if (!$u || ($u['role'] ?? '') !== $role) {
    http_response_code(403);
    echo "Forbidden";
    exit;
  }
}

/**
 * Visibility rule for companies:
 * - admin: all
 * - manager: all except private=1
 * - user: only initials match user's initials
 */
function companies_visibility_clause(?array $u): array {
  if (!$u) return ['0=1', []];

  $role = (string)($u['role'] ?? 'user');
  if ($role === 'admin') return ['1=1', []];
  if ($role === 'manager') return ['(private IS NULL OR private = 0)', []];

  $init = (string)($u['initials'] ?? '');
  return ['(initials = :vis_init)', [':vis_init' => $init]];
}

function can_access_company_row(array $companyRow, array $u): bool {
  $role = (string)($u['role'] ?? 'user');
  if ($role === 'admin') return true;
  if ($role === 'manager') return empty($companyRow['private']);
  return (string)($companyRow['initials'] ?? '') === (string)($u['initials'] ?? '');
}
