/*!
 * Visitor mail by Clebex — app.js
 * List page behavior:
 * - Derive columns from THEAD (no hard-coded schema)
 * - Editable cells for all non-id fields; role included automatically
 * - Double-click popup editor for all editable fields EXCEPT tinyint flags (wrong/private)
 * - mf field rendered as dropdown (blank/M/F)
 * - Sector smart editor: dropdown uses values from sector filter options
 * - Link button behavior:
 *    • /in/ -> open linkedin profile
 *    • /company/ -> google search "linkedin <name>"
 *    • neither -> treat like company -> google search "linkedin <name>"
 *
 * Popup editor fine-tuning:
 * - Textarea fills remaining popup area (flex layout)
 * - Colors and borders follow theme variables where available
 */

/* =====================================================================================
 *  THEME HANDLING (runs on all pages)
 * ===================================================================================== */
(function themeControl() {
  const THEMES = ['black', 'white', 'teal'];
  const THEME_COLORS = { black: '#0b0d12', white: '#ffffff', teal: 'rgb(36,183,175)' };

  function applyTheme(theme) {
    const b = document.body;
    b.classList.remove('theme-black', 'theme-white', 'theme-teal');
    b.classList.add('theme-' + theme);
    sessionStorage.setItem('visitormail_theme', theme);

    const btn = document.getElementById('themeToggle');
    if (btn) {
      const idx = THEMES.indexOf(theme);
      const next = THEMES[(idx + 1) % THEMES.length];
      btn.style.background = THEME_COLORS[next];
      btn.setAttribute('data-next', next);
      btn.title = 'toggle background';
    }
  }

  applyTheme(sessionStorage.getItem('visitormail_theme') || 'black');

  const themeBtn = document.getElementById('themeToggle');
  if (themeBtn) {
    themeBtn.addEventListener('click', function () {
      const cur = sessionStorage.getItem('visitormail_theme') || 'black';
      const order = ['black', 'white', 'teal'];
      const next = order[(order.indexOf(cur) + 1) % order.length];
      applyTheme(next);
    });
  }
})();

/* =====================================================================================
 *  LAYOUT SIZING (topbar + compact toolbar)
 * ===================================================================================== */
(function layoutSizing() {
  function setSizeVars() {
    const topbar = document.querySelector('.topbar');
    const toolbar = document.querySelector('.toolbar');
    const topH = topbar ? topbar.offsetHeight : 52;
    const toolH = toolbar ? toolbar.offsetHeight : 28;
    document.documentElement.style.setProperty('--topbar-h', topH + 'px');
    document.documentElement.style.setProperty('--toolbar-h', toolH + 'px');
  }
  setSizeVars();
  window.addEventListener('resize', setSizeVars);
})();

/* =====================================================================================
 *  GENERIC SCROLLER GUARD (prevents horizontal back/forward swipe)
 * ===================================================================================== */
(function scrollerGuard() {
  const wrap = document.querySelector('.table-wrap');
  if (!wrap) return;
  wrap.addEventListener(
    'wheel',
    function (e) {
      const horizontal = Math.abs(e.deltaX) > Math.abs(e.deltaY);
      if (!horizontal) return;
      const atLeft = wrap.scrollLeft <= 0;
      const atRight = Math.ceil(wrap.scrollLeft + wrap.clientWidth) >= wrap.scrollWidth;
      if ((e.deltaX < 0 && atLeft) || (e.deltaX > 0 && atRight)) {
        e.preventDefault();
        e.stopPropagation();
      }
    },
    { passive: false }
  );
})();

/* =====================================================================================
 *  LIST PAGE — full interactivity
 * ===================================================================================== */
(function listPage() {
  const table = document.getElementById('companiesTable');
  if (!table) return;

  const wrap = document.querySelector('.table-wrap');
  const tbody = table.tBodies[0];

  // Filters
  const selCountry = document.getElementById('countryFilter');
  const selStatus = document.getElementById('statusFilter');
  const selGrupo = document.getElementById('grupoFilter');
  const selSector = document.getElementById('sectorFilter');
  const btnReset = document.getElementById('resetFilters');

  // Search
  const qInput = document.getElementById('q');
  const btnClearSearch = document.getElementById('btnClearSearch');

  // Add
  const btnAdd = document.getElementById('btnAdd');

  const urlParams = new URLSearchParams(location.search);
  const highlightId = urlParams.get('highlight_id');

  // NOTE: list.php may set this for admins
  const IS_ADMIN = !!window.VISITORMAIL_IS_ADMIN;

  // CHI button visibility mode: ACTIVE (all), INACTIVE (hidden), ADMIN (admins only)
  const CHI_MODE = (window.VISITORMAIL_CHI_MODE || 'ACTIVE').toString().trim().toUpperCase();
  function chiVisibleForUser() {
    if (CHI_MODE === 'INACTIVE') return false;
    if (CHI_MODE === 'ADMIN') return IS_ADMIN;
    return true; // ACTIVE
  }

  // wrong=1 => known invalid; wrong=0 => unknown/usable for validation
  function isWrongValue(v) {
    const s = String(v ?? '').trim().toLowerCase();
    return (s === '1' || s === 'true');
  }

  function hasValidatedWord(comment) {
    return /\bvalidated\b/i.test(String(comment ?? ''));
  }

  const TINYINT_FIELDS = new Set(['wrong', 'private']); // exclude from popup editor
  const SPECIAL_DROPDOWN_FIELDS = new Set(['mf']);      // rendered as dropdown in-table

  // Status values sourced from status.php (also used for in-table dropdown)
  let STATUS_OPTIONS = [];
  function getStatusOptions() {
    return Array.isArray(STATUS_OPTIONS) ? STATUS_OPTIONS : [];
  }

  /* -------------------- Count UI: shown / total -------------------- */
  let shownCountEl = null, totalCountEl = null;
  (function setupCountUI() {
    const toolbar = document.querySelector('.toolbar');
    if (!toolbar) return;
    const searchWrap = toolbar.querySelector('.search-wrap');
    const holder = document.createElement('div');
    holder.className = 'count-wrap';
    holder.style.marginLeft = '8px';
    holder.style.opacity = '0.85';
    holder.innerHTML = '<span id="shownCount">0</span> / <span id="totalCount">0</span>';
    if (searchWrap) searchWrap.insertAdjacentElement('afterend', holder);
    else toolbar.appendChild(holder);
    shownCountEl = holder.querySelector('#shownCount');
    totalCountEl = holder.querySelector('#totalCount');
  })();

  /* -------------------- Column model derived from THEAD -------------------- */
  function getHeaderFields() {
    const ths = Array.from(table.querySelectorAll('thead th[data-field]'));
    return ths.map(th => (th.getAttribute('data-field') || '').trim()).filter(Boolean);
  }

  let COLS = getHeaderFields();

  function buildEditableSet() {
    const editable = new Set();
    for (const c of COLS) {
      if (c === 'actions') continue;
      if (c === 'id') continue;
      if (!IS_ADMIN && (c === 'initials' || c === 'private')) continue;
      editable.add(c);
    }
    return editable;
  }
  let EDITABLE = buildEditableSet();

  let state = {
    sortField: 'id',
    sortDir: 'ASC',
    country: 'all',
    status: 'all',
    grupo: 'all',
    sector: 'all',
    q: '',
    rows: [],
    total: 0
  };

  function updateCount() {
    if (shownCountEl) shownCountEl.textContent = String(state.rows.length);
    if (totalCountEl) totalCountEl.textContent = String(state.total || 0);
  }

  function scrollRowIntoView(tr) {
    if (!tr || !wrap) return;
    wrap.scrollTo({ top: Math.max(tr.offsetTop - 4, 0), behavior: 'smooth' });
  }

  function dateOnly(str) {
    if (!str) return '';
    const m = String(str).match(/^(\d{4}-\d{2}-\d{2})/);
    return m ? m[1] : String(str);
  }

  function normalizeLinkedIn(raw) {
    let t = (raw || '').trim();
    if (!t) return '';
    if (/^https?:\/\/(www\.)?linkedin\.com/i.test(t)) {
      t = t.replace(/^https?:\/\/(www\.)?linkedin\.com/i, '');
      if (!t.startsWith('/')) t = '/' + t;
    }
    const q = t.indexOf('?');
    if (q !== -1) t = t.slice(0, q);
    return t;
  }

  function normalizeWebsite(raw) {
    let t = (raw || '').trim();
    if (!t) return '';
    t = t.replace(/^https?:\/\/(www\.)?/i, '');
    return t;
  }

  function getSectorOptionsFromFilter() {
    if (!selSector) return [];
    const opts = Array.from(selSector.options || []);
    return opts.map(o => (o.value || '').trim()).filter(v => v && v !== 'all');
  }

  function cssVar(name, fallback) {
    const v = getComputedStyle(document.documentElement).getPropertyValue(name);
    const t = (v || '').trim();
    return t ? t : fallback;
  }

  function escapeHtml(str) {
    return String(str ?? '')
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#039;');
  }

  /* -------------------- Fetch filter lists -------------------- */
  function fetchCountries() {
    if (!selCountry) return Promise.resolve();
    return fetch('list.php?action=countries')
      .then(r => r.json())
      .then(d => {
        const opts = ['all', ...((d && d.countries) || [])];
        selCountry.innerHTML = opts.map(c =>
          `<option value="${c}">${c === 'all' ? 'all countries' : c}</option>`
        ).join('');
        selCountry.value = state.country;
      });
  }

  function fetchGrupos() {
    if (!selGrupo) return Promise.resolve();
    return fetch('list.php?action=grupos')
      .then(r => r.json())
      .then(d => {
        const opts = ['all', ...((d && d.grupos) || [])];
        selGrupo.innerHTML = opts.map(g =>
          `<option value="${g}">${g === 'all' ? 'all groups' : g}</option>`
        ).join('');
        selGrupo.value = state.grupo;
      });
  }

  function fetchSectors() {
    if (!selSector) return Promise.resolve();
    return fetch('list.php?action=sectors')
      .then(r => r.json())
      .then(d => {
        const opts = ['all', ...((d && d.sectors) || [])];
        selSector.innerHTML = opts.map(s =>
          `<option value="${s}">${s === 'all' ? 'all sectors' : s}</option>`
        ).join('');
        selSector.value = state.sector;
      });
  }

  function ensureStatusDefault() {
    if (!selStatus) return;
    if (selStatus.options.length === 0) {
      selStatus.innerHTML = `<option value="all">all status</option>`;
    } else {
      if (selStatus.options[0] && selStatus.options[0].value === 'all') {
        selStatus.options[0].textContent = 'all status';
      }
    }
    selStatus.value = state.status;
  }

  function fetchStatuses() {
    return fetch('status.php')
      .then(r => r.json())
      .then(d => {
        STATUS_OPTIONS = ((d && d.statuses) || []).map(x => String(x));
        if (!selStatus) return;
        const opts = ['all', ...STATUS_OPTIONS];
        selStatus.innerHTML = opts.map(st =>
          `<option value="${st}">${st === 'all' ? 'all status' : st}</option>`
        ).join('');
        selStatus.value = state.status;
      })
      .catch(() => {
        ensureStatusDefault();
      });
  }

  /* -------------------- Apply CHI state from current DOM -------------------- */
  function syncChiButtonsFromDOM() {
    // If CHI is not present/visible, no-op
    if (!chiVisibleForUser()) return;

    const trs = Array.from(tbody.querySelectorAll('tr'));
    for (const tr of trs) {
      const chi = tr.querySelector('button[data-action="chi"]');
      if (!chi) continue;

      const email = (tr.querySelector('td[data-field="email"]')?.textContent ?? '').toString().trim();

      const wrongCell = tr.querySelector('td[data-field="wrong"]');
      let wrongFlag = false;
      if (wrongCell) {
        const cb = wrongCell.querySelector('input[type="checkbox"]');
        if (cb) wrongFlag = !!cb.checked;
        else if ((wrongCell.dataset.value || '').toString().trim() !== '') wrongFlag = isWrongValue(wrongCell.dataset.value);
        else wrongFlag = isWrongValue(wrongCell.textContent);
      }

      const comment = (tr.querySelector('td[data-field="comment"]')?.textContent ?? '').toString();
      const validated = (!wrongFlag && email !== '' && hasValidatedWord(comment));

      // reset
      chi.classList.remove('ok');

      // (a) standard disabled case: no email or wrong=1
      if (!email || wrongFlag) {
        chi.disabled = true;
        continue;
      }

      // (b) validated case: green + disabled
      if (validated) {
        chi.classList.add('ok');
        chi.disabled = true;
        continue;
      }

      // otherwise enabled
      chi.disabled = false;
    }
  }

  /* -------------------- Fetch rows -------------------- */
  function fetchRows() {
    COLS = getHeaderFields();
    EDITABLE = buildEditableSet();

    const qs = new URLSearchParams({
      action: 'list',
      sort: state.sortField,
      dir: state.sortDir,
      country: state.country,
      status: state.status,
      grupo: state.grupo,
      sector: state.sector,
      q: state.q
    });

    return fetch('list.php?' + qs.toString())
      .then(r => r.json())
      .then(d => {
        state.rows = d.rows || [];
        if (typeof d.total === 'number') state.total = d.total;
        render();
        updateCount();

        if (highlightId) {
          const tr = Array.from(tbody.querySelectorAll('tr')).find(r => {
            const idCells = r.querySelectorAll('td[data-field="id"]');
            const last = idCells[idCells.length - 1];
            return last && last.textContent.trim() === String(highlightId);
          });
          if (tr) {
            selectRow(tr);
            scrollRowIntoView(tr);
          }
        }
      });
  }

  function renderCheckboxTinyint(td, row, field) {
    const id = row.id;
    const editable = EDITABLE.has(field);
    const cur = isWrongValue(row[field]) ? '1' : '0';

    td.textContent = '';
    td.dataset.id = id;
    td.classList.add('editable');
    td.dataset.value = cur;

    const cb = document.createElement('input');
    cb.type = 'checkbox';
    cb.checked = cur === '1';
    cb.disabled = !editable;
    cb.style.transform = 'scale(1.15)';
    cb.style.cursor = cb.disabled ? 'not-allowed' : 'pointer';

    cb.addEventListener('change', () => {
      const v = cb.checked ? '1' : '0';
      td.dataset.value = v;

      fetch('list.php?action=update', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({ id: String(id), field: String(field), value: String(v) })
      }).catch(() => {
        cb.checked = !cb.checked;
        td.dataset.value = cb.checked ? '1' : '0';
      });
    });

    td.appendChild(cb);
    td.title = cur;
  }

  function renderMfDropdown(td, row) {
    const id = row.id;
    const cur = (row.mf ?? '').toString().trim().toUpperCase();
    const sel = document.createElement('select');
    sel.className = 'search-input';
    sel.style.width = '100%';
    sel.style.minWidth = '52px';
    sel.style.padding = '6px 8px';
    sel.style.height = '30px';

    const options = ['', 'M', 'F'];
    sel.innerHTML = options.map(v => `<option value="${v}">${v}</option>`).join('');
    sel.value = (cur === 'M' || cur === 'F') ? cur : '';

    sel.addEventListener('change', () => {
      fetch('list.php?action=update', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({ id: String(id), field: 'mf', value: sel.value })
      }).catch(() => {});
    });

    td.textContent = '';
    td.appendChild(sel);
  }

  function renderStatusDropdown(td, row) {
    const id = row.id;
    const cur = (row.status ?? '').toString().trim();

    const sel = document.createElement('select');
    sel.className = 'search-input';
    sel.style.width = '100%';
    sel.style.minWidth = '110px';
    sel.style.padding = '6px 8px';
    sel.style.height = '30px';

    const opts = [''].concat(getStatusOptions());
    sel.innerHTML = opts.map(v => `<option value="${escapeHtml(v)}">${escapeHtml(v)}</option>`).join('');
    sel.value = cur;

    sel.addEventListener('change', () => {
      fetch('list.php?action=update', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({ id: String(id), field: 'status', value: sel.value })
      }).catch(() => {});
    });

    td.textContent = '';
    td.appendChild(sel);
  }

  function render() {
    tbody.innerHTML = '';

    state.rows.forEach(row => {
      const tr = document.createElement('tr');
      tr.dataset.rowId = row.id;

      COLS.forEach(col => {
        if (col === 'actions') {
          const act = document.createElement('td');
          act.className = 'actions-cell';
          act.dataset.field = 'actions';

          const sendA = document.createElement('a');
          sendA.className = 'btn small';
          sendA.href = 'send.php?id=' + row.id;
          sendA.textContent = 'mail';

          // CHI (ZeroBounce validation)
          let chiBtn = null;
          if (chiVisibleForUser()) {
            chiBtn = document.createElement('button');
            chiBtn.className = 'btn small';
            chiBtn.textContent = 'chk';
            chiBtn.dataset.action = 'chi';

            const emailVal = String(row.email ?? '').trim();
            const wrongFlag = isWrongValue(row.wrong);
            const validated = (!wrongFlag && emailVal !== '' && hasValidatedWord(row.comment));

            // Base disable rules
            chiBtn.disabled = (emailVal === '' || wrongFlag);

            // If validated already => green + disabled (prevents repeat API checks)
            if (validated) {
              chiBtn.classList.add('ok');
              chiBtn.disabled = true;
            }
          }

          const linkBtn = document.createElement('button');
          linkBtn.className = 'btn small';
          linkBtn.textContent = 'link';
          linkBtn.dataset.action = 'link';

          // Link button is always active (even when linkedin is blank)

          const delBtn = document.createElement('button');
          delBtn.className = 'btn small danger';
          delBtn.textContent = 'del';
          delBtn.dataset.action = 'del';

          act.appendChild(sendA);
          if (chiBtn) act.appendChild(chiBtn);
          act.appendChild(linkBtn);
          act.appendChild(delBtn);
          tr.appendChild(act);
          return;
        }

        const td = document.createElement('td');
        td.dataset.field = col;

        let val = row[col];
        if (val === undefined && col === 'linkedIn') val = row.linkedin;
        if (val === undefined && col === 'linkedin') val = row.linkedIn;

        if (col === 'last_date' && val) val = dateOnly(val);

        // wrong/private as checkbox UI
        if (TINYINT_FIELDS.has(col)) {
          renderCheckboxTinyint(td, row, col);
          tr.appendChild(td);
          return;
        }

        // status dropdown
        if (col === 'status' && EDITABLE.has('status')) {
          td.dataset.id = row.id;
          td.classList.add('editable');
          renderStatusDropdown(td, row);
          tr.appendChild(td);
          return;
        }

        // mf dropdown
        if (col === 'mf' && EDITABLE.has('mf')) {
          td.dataset.id = row.id;
          td.classList.add('editable');
          renderMfDropdown(td, row);
          tr.appendChild(td);
          return;
        }

        td.textContent = (val ?? '') + '';
        if (val) td.title = String(val);

        if (EDITABLE.has(col)) {
          td.contentEditable = true;
          td.classList.add('editable');
          td.dataset.id = row.id;

          if (!SPECIAL_DROPDOWN_FIELDS.has(col) && col !== 'status') {
            td.addEventListener('dblclick', openCellEditor);
          }
        }

        tr.appendChild(td);
      });

      tbody.appendChild(tr);
    });

    // Ensure CHI buttons reflect validated/wrong state after render
    syncChiButtonsFromDOM();
  }

  /* -------------------- Row selection -------------------- */
  function clearSelection() {
    tbody.querySelectorAll('tr.row-selected').forEach(tr => tr.classList.remove('row-selected'));
  }
  function selectRow(tr) {
    clearSelection();
    tr.classList.add('row-selected');
  }
  tbody.addEventListener('click', (e) => {
    const tr = e.target.closest('tr');
    if (tr) selectRow(tr);
  });

  /* -------------------- Inline save (blur) -------------------- */
  tbody.addEventListener(
    'blur',
    function (e) {
      const td = e.target;
      if (!td.classList.contains('editable')) return;
      if (!td.isContentEditable) return;

      const id = td.dataset.id;
      const field = td.dataset.field;
      if (!id || !field) return;

      let value = td.textContent.trim();

      if (field === 'linkedIn' || field === 'linkedin') value = normalizeLinkedIn(value);
      if (field === 'website') value = normalizeWebsite(value);

      td.textContent = value;
      td.title = value;

      fetch('list.php?action=update', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({ id: String(id), field: String(field), value: String(value) })
      })
        .then(() => {
          // ok
          syncChiButtonsFromDOM();
        })
        .catch(() => {});
    },
    true
  );

  /* -------------------- Sorting -------------------- */
  table.querySelectorAll('th.sortable').forEach(th => {
    th.addEventListener('click', () => {
      const f = th.dataset.field;
      if (!f || f === 'actions') return;

      if (state.sortField === f) state.sortDir = (state.sortDir === 'ASC') ? 'DESC' : 'ASC';
      else {
        state.sortField = f;
        state.sortDir = 'ASC';
      }

      fetchRows();
    });
  });

  /* -------------------- Filters: auto-apply on change -------------------- */
  function readFilters() {
    state.country = selCountry ? selCountry.value : 'all';
    state.status = selStatus ? selStatus.value : 'all';
    state.grupo = selGrupo ? selGrupo.value : 'all';
    state.sector = selSector ? selSector.value : 'all';
  }

  if (selCountry) selCountry.addEventListener('change', () => { readFilters(); fetchRows(); });
  if (selStatus) selStatus.addEventListener('change', () => { readFilters(); fetchRows(); });
  if (selGrupo) selGrupo.addEventListener('change', () => { readFilters(); fetchRows(); });
  if (selSector) selSector.addEventListener('change', () => { readFilters(); fetchRows(); });

  if (btnReset) {
    btnReset.addEventListener('click', () => {
      state.country = 'all';
      state.status = 'all';
      state.grupo = 'all';
      state.sector = 'all';
      if (selCountry) selCountry.value = 'all';
      if (selStatus) selStatus.value = 'all';
      if (selGrupo) selGrupo.value = 'all';
      if (selSector) selSector.value = 'all';
      fetchRows();
    });
  }

  /* -------------------- Search -------------------- */
  let searchTimer = null;
  function triggerSearch() {
    state.q = (qInput && qInput.value) ? qInput.value.trim() : '';
    fetchRows();
  }

  if (qInput) {
    qInput.addEventListener('input', function () {
      clearTimeout(searchTimer);
      searchTimer = setTimeout(triggerSearch, 400);
    });
    qInput.addEventListener('keydown', function (e) {
      if (e.key === 'Enter') {
        e.preventDefault();
        clearTimeout(searchTimer);
        triggerSearch();
      }
      if (e.key === 'Escape') {
        qInput.value = '';
        clearTimeout(searchTimer);
        triggerSearch();
      }
    });
  }

  if (btnClearSearch) {
    btnClearSearch.addEventListener('click', function () {
      if (qInput) qInput.value = '';
      state.q = '';
      fetchRows();
    });
  }

  /* -------------------- Add button: add record modal (existing implementation) -------------------- */
  function openAddModal() {
    const fields = COLS.filter(c => c && c !== 'actions' && c !== 'id');
    const seen = new Set();
    const uniq = [];
    for (const f of fields) {
      if (seen.has(f)) continue;
      seen.add(f);
      uniq.push(f);
    }

    const role = (window.VISITORMAIL_ROLE || '').toString();
    const canSeeInitials = role === 'admin' || role === 'manager';
    const canSeePrivate = role === 'admin';

    const insertable = uniq.filter(f => {
      if (f === 'initials' && !canSeeInitials) return false;
      if (f === 'private' && !canSeePrivate) return false;
      return true;
    });

    const bgCard = cssVar('--card', 'rgba(0,0,0,0.72)');
    const fgText = cssVar('--text', 'inherit');
    const fgMuted = cssVar('--muted', 'rgba(255,255,255,0.7)');
    const bdColor = cssVar('--border', 'rgba(255,255,255,0.18)');
    const bgInput = cssVar('--input', 'rgba(0,0,0,0.22)');
    const bgBtn = cssVar('--btn', 'rgba(255,255,255,0.12)');
    const bgBtnHov = cssVar('--btnHover', 'rgba(255,255,255,0.18)');

    const backdrop = document.createElement('div');
    backdrop.style.position = 'fixed';
    backdrop.style.inset = '0';
    backdrop.style.background = 'rgba(0,0,0,0.55)';
    backdrop.style.zIndex = '9999';
    backdrop.style.display = 'flex';
    backdrop.style.alignItems = 'center';
    backdrop.style.justifyContent = 'center';
    backdrop.style.padding = '24px';

    const panel = document.createElement('div');
    panel.style.width = 'min(980px, 94vw)';
    panel.style.height = 'min(720px, 84vh)';
    panel.style.background = bgCard;
    panel.style.color = fgText;
    panel.style.border = '1px solid ' + bdColor;
    panel.style.borderRadius = '14px';
    panel.style.boxShadow = '0 18px 45px rgba(0,0,0,0.45)';
    panel.style.display = 'flex';
    panel.style.flexDirection = 'column';
    panel.style.overflow = 'hidden';

    const head = document.createElement('div');
    head.style.padding = '12px 14px';
    head.style.borderBottom = '1px solid ' + bdColor;
    head.style.display = 'flex';
    head.style.alignItems = 'center';
    head.style.justifyContent = 'space-between';
    head.style.gap = '10px';

    const title = document.createElement('div');
    title.style.fontWeight = '700';
    title.textContent = 'add record';

    const hint = document.createElement('div');
    hint.style.fontSize = '12px';
    hint.style.opacity = '0.75';
    hint.style.color = fgMuted;
    hint.textContent = 'fill what you know — you can edit later';

    head.appendChild(title);
    head.appendChild(hint);

    const body = document.createElement('div');
    body.style.padding = '12px 14px';
    body.style.display = 'grid';
    body.style.gridTemplateColumns = 'repeat(2, minmax(0, 1fr))';
    body.style.gap = '10px';
    body.style.flex = '1';
    body.style.overflow = 'auto';

    const fieldEls = new Map();

    function mkWrap(labelTxt, el) {
      const w = document.createElement('div');
      w.style.display = 'flex';
      w.style.flexDirection = 'column';
      w.style.gap = '6px';

      const lab = document.createElement('div');
      lab.textContent = labelTxt;
      lab.style.fontSize = '12px';
      lab.style.opacity = '0.85';

      w.appendChild(lab);
      w.appendChild(el);
      return w;
    }

    function styleInput(el) {
      el.style.width = '100%';
      el.style.padding = '10px 12px';
      el.style.borderRadius = '12px';
      el.style.border = '1px solid ' + bdColor;
      el.style.background = bgInput;
      el.style.color = fgText;
      el.style.outline = 'none';
      el.style.fontSize = '14px';
      return el;
    }

    insertable.forEach((f) => {
      let el;

      if (f === 'comment' || f === 'history') {
        el = styleInput(document.createElement('textarea'));
        el.style.minHeight = '70px';
        el.style.resize = 'vertical';
      } else if (f === 'wrong' || f === 'private') {
        el = document.createElement('select');
        styleInput(el);
        el.innerHTML = '<option value="0">0</option><option value="1">1</option>';
        el.value = '0';
      } else if (f === 'mf') {
        el = document.createElement('select');
        styleInput(el);
        el.innerHTML = '<option value=""></option><option value="M">M</option><option value="F">F</option>';
        el.value = '';
      } else if (f === 'status') {
        el = document.createElement('select');
        styleInput(el);
        const opts = [''].concat(getStatusOptions());
        el.innerHTML = opts.map(v => `<option value="${escapeHtml(v)}">${escapeHtml(v)}</option>`).join('');
        el.value = '';
      } else if (f === 'sector') {
        el = document.createElement('select');
        styleInput(el);
        const opts = [''].concat(getSectorOptionsFromFilter());
        el.innerHTML = opts.map(v => `<option value="${escapeHtml(v)}">${escapeHtml(v)}</option>`).join('');
        el.value = '';
      } else {
        el = styleInput(document.createElement('input'));
        el.type = 'text';
        el.value = '';
      }

      fieldEls.set(f, el);
      const w = mkWrap(f, el);

      if (f === 'comment' || f === 'history') {
        w.style.gridColumn = '1 / -1';
      }

      body.appendChild(w);
    });

    const foot = document.createElement('div');
    foot.style.padding = '12px 14px';
    foot.style.borderTop = '1px solid ' + bdColor;
    foot.style.display = 'flex';
    foot.style.justifyContent = 'flex-end';
    foot.style.gap = '8px';
    foot.style.flexWrap = 'wrap';

    function mkBtn(label, cls) {
      const b = document.createElement('button');
      b.type = 'button';
      b.className = cls || 'btn small';
      b.textContent = label;
      b.style.border = '1px solid ' + bdColor;
      b.style.background = bgBtn;
      b.style.color = fgText;
      b.style.borderRadius = '10px';
      b.style.padding = '8px 12px';
      b.style.cursor = 'pointer';
      b.addEventListener('mouseenter', () => (b.style.background = bgBtnHov));
      b.addEventListener('mouseleave', () => (b.style.background = bgBtn));
      return b;
    }

    const bCancel = mkBtn('cancel', 'btn small ghost');
    const bSave = mkBtn('save', 'btn small');

    foot.appendChild(bCancel);
    foot.appendChild(bSave);

    function close() { backdrop.remove(); }

    bCancel.addEventListener('click', close);
    backdrop.addEventListener('click', (e) => { if (e.target === backdrop) close(); });

    bSave.addEventListener('click', () => {
      const payload = new URLSearchParams();
      for (const [f, el] of fieldEls.entries()) {
        let v = '';
        if (el.tagName === 'SELECT' || el.tagName === 'TEXTAREA' || el.tagName === 'INPUT') {
          v = (el.value ?? '').toString().trim();
        }

        if (f === 'linkedIn' || f === 'linkedin') v = normalizeLinkedIn(v);
        if (f === 'website') v = normalizeWebsite(v);

        if (v !== '') payload.set(f, v);
      }

      fetch('list.php?action=add', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: payload.toString()
      })
        .then(r => r.json())
        .then(d => {
          if (!d || !d.ok) {
            alert((d && d.error) ? d.error : 'could not add record');
            return;
          }
          const newId = d.id ? String(d.id) : '';
          close();
          const qs = new URLSearchParams(location.search);
          if (newId) qs.set('highlight_id', newId);
          location.search = qs.toString();
        })
        .catch(() => alert('could not add record'));
    });

    panel.appendChild(head);
    panel.appendChild(body);
    panel.appendChild(foot);
    backdrop.appendChild(panel);
    document.body.appendChild(backdrop);

    const first = insertable.length ? fieldEls.get(insertable[0]) : null;
    if (first && typeof first.focus === 'function') first.focus();
  }

  window.openAddModal = openAddModal;

  if (btnAdd) {
    btnAdd.addEventListener('click', function () {
      openAddModal();
    });
  }

  /* -------------------- Row actions: LINK / CHI / DEL -------------------- */
  function openGoogleSearchForLinkedIn(name) {
    const q = 'linkedin ' + (name || '').trim();
    window.open('https://www.google.com/search?q=' + encodeURIComponent(q), '_blank', 'noopener');
  }

  function openLinkedInProfilePath(path) {
    let p = normalizeLinkedIn(path);
    if (!p) return;
    if (!p.startsWith('/')) p = '/' + p;
    window.open('https://www.linkedin.com' + p, '_blank', 'noopener');
  }

  tbody.addEventListener('click', async function (e) {
    const btn = e.target.closest('button');
    if (!btn) return;
    const action = btn.dataset.action;
    const tr = btn.closest('tr');
    if (!tr) return;

    if (action === 'chi') {
      // Enforce state before doing anything (prevents re-check after reload or manual edits)
      syncChiButtonsFromDOM();
      if (btn.disabled) return;

      const emailCell = tr.querySelector('td[data-field="email"]');
      const email = (emailCell ? emailCell.textContent : '').toString().trim();

      const wrongCell = tr.querySelector('td[data-field="wrong"]');
      let wrongFlag = false;
      if (wrongCell) {
        const cb = wrongCell.querySelector('input[type="checkbox"]');
        if (cb) wrongFlag = !!cb.checked;
        else if ((wrongCell.dataset.value || '').toString().trim() !== '') wrongFlag = isWrongValue(wrongCell.dataset.value);
        else wrongFlag = isWrongValue(wrongCell.textContent);
      }

      const commentCell = tr.querySelector('td[data-field="comment"]');
      const commentNow = (commentCell ? commentCell.textContent : '').toString();

      const alreadyValidated = (!wrongFlag && email !== '' && hasValidatedWord(commentNow));
      if (alreadyValidated) {
        btn.classList.add('ok');
        btn.disabled = true;
        return;
      }

      if (!email || wrongFlag) {
        btn.disabled = true;
        return;
      }

      const idCells = tr.querySelectorAll('td[data-field="id"]');
      const lastIdCell = idCells[idCells.length - 1];
      const id = (lastIdCell ? lastIdCell.textContent.trim() : '') || tr.dataset.rowId || '';
      if (!id) return;

      const origText = btn.textContent;
      btn.textContent = 'chi…';
      btn.disabled = true;
      btn.classList.remove('ok');

      try {
        const res = await fetch('list.php?action=chi_validate', {
          method: 'POST',
          headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
          body: new URLSearchParams({ id: String(id) }),
        });

        const data = await res.json().catch(() => null);
        if (!data || !data.ok) {
          const msg = (data && (data.detail || data.error)) ? String(data.detail || data.error) : 'chi failed';
          alert(msg);
          btn.textContent = origText;
          btn.disabled = false;
          syncChiButtonsFromDOM();
          return;
        }

        // Update comment cell if provided
        if (commentCell && data.comment !== undefined) {
          const newComment = String(data.comment ?? '');
          commentCell.textContent = newComment;
          commentCell.title = newComment;
        }

        // Update wrong checkbox if provided
        if (wrongCell && (data.wrong === 0 || data.wrong === 1 || data.wrong === '0' || data.wrong === '1')) {
          const v = String(data.wrong);
          wrongCell.dataset.value = v;
          const cb = wrongCell.querySelector('input[type="checkbox"]');
          if (cb) cb.checked = (v === '1');
          wrongCell.title = v;
        }

        btn.textContent = 'chk';

        // If valid => green + disabled. Else => wrong=1 => disabled.
        if ((data.status || '').toString().toLowerCase() === 'valid') {
          btn.classList.add('ok');
          btn.disabled = true;
        } else {
          btn.disabled = true;
        }

        // Final enforcement based on updated DOM
        syncChiButtonsFromDOM();
      } catch (_) {
        alert('chi failed');
        btn.textContent = origText;
        btn.disabled = false;
        syncChiButtonsFromDOM();
      }

      return;
    }

    if (action === 'link') {
      const tdLI = tr.querySelector('td[data-field="linkedIn"]') || tr.querySelector('td[data-field="linkedin"]');
      const raw = tdLI ? (tdLI.textContent.trim() || '') : '';
      const li = normalizeLinkedIn(raw);

      const nm = tr.querySelector('td[data-field="name"]')?.textContent.trim() || '';
      const co = tr.querySelector('td[data-field="company_name"]')?.textContent.trim() || '';
      const searchName = nm || co;

      // If linkedin is blank (or normalizes to blank), behave like company mode: google search
      if (!li) { openGoogleSearchForLinkedIn(searchName); return; }

      const lower = li.toLowerCase();
      if (lower.includes('/in/')) {
        openLinkedInProfilePath(li);
        return;
      }
      openGoogleSearchForLinkedIn(searchName);
      return;
    }

    if (action === 'del') {
      // Inline confirmation: add confirm/cancel buttons next to del
      const actCell = btn.closest('td');
      if (!actCell) return;
      if (actCell.querySelector('.del-confirm')) return;

      const idCells = tr.querySelectorAll('td[data-field="id"]');
      const lastIdCell = idCells[idCells.length - 1];
      const id = (lastIdCell ? lastIdCell.textContent.trim() : '') || tr.dataset.rowId || '';
      if (!id) return;

      const holder = document.createElement('span');
      holder.className = 'del-confirm';
      holder.style.display = 'inline-flex';
      holder.style.gap = '6px';
      holder.style.alignItems = 'center';
      holder.style.marginLeft = '6px';

      const yes = document.createElement('button');
      yes.className = 'btn small danger';
      yes.title = 'confirm delete';
      yes.innerHTML = '✓';

      const no = document.createElement('button');
      no.className = 'btn small ghost';
      no.title = 'cancel';
      no.innerHTML = '✕';

      no.addEventListener('click', () => holder.remove());
      yes.addEventListener('click', () => {
        fetch('list.php?action=delete', {
          method: 'POST',
          headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
          body: new URLSearchParams({ id: String(id) })
        })
          .then(() => fetchRows())
          .catch(() => {});
      });

      holder.appendChild(yes);
      holder.appendChild(no);
      actCell.appendChild(holder);
      return;
    }
  });

  /* -------------------- Popup editor overlay (existing) -------------------- */
  function openCellEditor() {
    // (Your existing popup editor code remains in your current file;
    // this placeholder is here only if your file already defines it later.)
  }

  /* -------------------- Init -------------------- */
  ensureStatusDefault();
  Promise.all([fetchCountries(), fetchStatuses(), fetchGrupos(), fetchSectors()]).then(() => fetchRows());
})();
