<?php
declare(strict_types=1);

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

require_login();
$me = current_user();
if (!$me) {
  header('Location: login.php');
  exit;
}

$roleUser = (string)($me['role'] ?? 'user');
$is_admin = ($roleUser === 'admin');

/* -------------------- Safe, namespaced helpers (no redeclare) -------------------- */
if (!function_exists('tm_h')) {
  function tm_h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}

if (!function_exists('tm_normalize_initials')) {
  function tm_normalize_initials(string $initials): string {
    $s = strtoupper(trim($initials));
    $s = preg_replace('/[^A-Z]/', '', $s); // keep only A-Z
    return $s ?? '';
  }
}

if (!function_exists('tm_initials_valid')) {
  function tm_initials_valid(string $initials): bool {
    return (bool)preg_match('/^[A-Z]{2,3}$/', $initials);
  }
}

if (!function_exists('tm_norm_bool01')) {
  function tm_norm_bool01($v): string {
    $t = strtoupper(trim((string)$v));
    return ($t === '1' || $t === 'Y' || $t === 'YES' || $t === 'TRUE') ? '1' : '0';
  }
}

if (!function_exists('tm_normalize_header_key')) {
  function tm_normalize_header_key(string $raw): string {
    $k = trim($raw);
    $k = preg_replace('/^\xEF\xBB\xBF/', '', $k); // strip BOM
    $k = strtolower($k);
    $k = str_replace([' ', '-', '.', '/', '\\'], '_', $k);
    $k = preg_replace('/_+/', '_', $k);

    $aliases = [
      'linkedin'      => 'linkedin',
      'linkedin_url'  => 'linkedin',
      'generalemail'  => 'general_email',
      'hqcity'        => 'hq_city',
      'lastdate'      => 'last_date',
      'lasttemplate'  => 'last_template',
      'm_f'           => 'mf',
      'm__f'          => 'mf',
      'mf'            => 'mf',
      'job'           => 'role', // legacy header mapping
    ];

    return $aliases[$k] ?? $k;
  }
}

if (!function_exists('tm_normalize_last_date')) {
  function tm_normalize_last_date($val): ?string {
    $v = trim((string)$val);
    if ($v === '') return null;

    if (preg_match('/^\d{4}-\d{2}-\d{2}(\s+\d{2}:\d{2}(:\d{2})?)?$/', $v)) {
      if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $v)) return $v . ' 00:00:00';
      if (preg_match('/^\d{4}-\d{2}-\d{2}\s+\d{2}:\d{2}$/', $v)) return $v . ':00';
      return $v;
    }

    if (preg_match('/^(\d{2})\.(\d{2})\.(\d{4})(\s+(\d{2}):(\d{2}))?$/', $v, $m)) {
      $dd = $m[1]; $mm = $m[2]; $yy = $m[3];
      $hh = $m[5] ?? '00';
      $mi = $m[6] ?? '00';
      return "{$yy}-{$mm}-{$dd} {$hh}:{$mi}:00";
    }

    return $v;
  }
}

if (!function_exists('tm_linkedin_from_company')) {
  function tm_linkedin_from_company(string $companyName): ?string {
    $companyName = trim($companyName);
    if ($companyName === '') return null;

    $s = mb_strtolower($companyName, 'UTF-8');
    $s = str_replace(["\t", "\r", "\n"], ' ', $s);
    $s = preg_replace("/[’']/u", '', $s);
    $s = preg_replace('/[^a-z0-9]+/u', ' ', $s);
    $s = preg_replace('/\s+/', ' ', trim($s));
    if ($s === '') return null;

    $slug = str_replace(' ', '-', $s);
    $slug = trim($slug, '-');
    if ($slug === '') return null;

    return "/company/{$slug}/";
  }
}

if (!function_exists('tm_clean_value')) {
  function tm_clean_value(string $canonKey, $raw) {
    $v = trim((string)$raw);

    if ($canonKey === 'mf') {
      if ($v === '') return null;
      $c = strtoupper(substr($v, 0, 1));
      return ($c === 'M' || $c === 'F') ? $c : null;
    }

    if ($canonKey === 'linkedin') {
      if ($v === '') return null;

      if (preg_match('~^https?://(www\.)?linkedin\.com~i', $v)) {
        $v = preg_replace('~^https?://(www\.)?linkedin\.com~i', '', $v);
        if ($v !== '' && $v[0] !== '/') $v = '/' . $v;
      }
      $q = strpos($v, '?');
      if ($q !== false) $v = substr($v, 0, $q);

      return ($v === '' ? null : $v);
    }

    if ($canonKey === 'website') {
      if ($v === '') return null;
      $v = preg_replace('~^https?://(www\.)?~i', '', $v);
      return ($v === '' ? null : $v);
    }

    if ($canonKey === 'wrong' || $canonKey === 'private') {
      return tm_norm_bool01($v);
    }

    if ($canonKey === 'last_date') {
      return tm_normalize_last_date($v);
    }

    return ($v === '' ? null : $v);
  }
}

/* -------------------- NEW: normalize company_name + name for duplicate test -------------------- */
if (!function_exists('tm_norm_name_key')) {
  function tm_norm_name_key(string $company, string $name): string {
    $c = trim(mb_strtolower($company, 'UTF-8'));
    $n = trim(mb_strtolower($name, 'UTF-8'));
    $c = preg_replace('/\s+/', ' ', $c) ?? $c;
    $n = preg_replace('/\s+/', ' ', $n) ?? $n;
    return $c . '||' . $n;
  }
}

/* -------------------- Configuration: application fields -------------------- */
$canonToDb = [
  'country'        => 'country',
  'language'       => 'language',
  'company_name'   => 'company_name',
  'sector'         => 'sector',
  'website'        => 'website',
  'name'           => 'name',
  'mf'             => 'mf',
  'role'           => 'role',
  'linkedin'       => 'linkedIn',
  'general_email'  => 'general_email',
  'hq_city'        => 'hq_city',
  'email'          => 'email',
  'comment'        => 'comment',
  'history'        => 'history',
  'status'         => 'status',
  'grupo'          => 'grupo',
  'wrong'          => 'wrong',
  'last_date'      => 'last_date',
  'last_template'  => 'last_template',
];
if ($is_admin) $canonToDb['private'] = 'private';

$expectedCanon = [
  'country','language','company_name','sector','website','name','mf','role','linkedin',
  'general_email','hq_city','email','grupo'
];

/* -------------------- Visibility + counts (visible rows) -------------------- */
$paramsVis = [];
$visClause = '1=1';
if (function_exists('companies_visibility_clause')) {
  [$visClause, $paramsVis] = companies_visibility_clause($me);
}

$stCnt = $pdo->prepare("SELECT COUNT(*) FROM companies WHERE $visClause");
$stCnt->execute($paramsVis);
$beforeCount = (int)$stCnt->fetchColumn();
$afterCount = $beforeCount;

/* -------------------- Admin initials list for dropdown -------------------- */
$userInitialsList = [];
if ($is_admin) {
  try {
    $rows = $pdo->query("SELECT DISTINCT initials FROM users WHERE initials IS NOT NULL AND initials<>'' ORDER BY initials ASC")
                ->fetchAll(PDO::FETCH_COLUMN);
    foreach ($rows as $ini) {
      $ini = tm_normalize_initials((string)$ini);
      if ($ini !== '' && !in_array($ini, $userInitialsList, true)) $userInitialsList[] = $ini;
    }
  } catch (Throwable $e) {
    $userInitialsList = [];
  }
}

$message = '';
$details = '';
$testReport = null;

function tm_detect_delim(string $tmpPath): string {
  $first = '';
  $fhTest = fopen($tmpPath, 'r');
  if ($fhTest) { $first = (string)fgets($fhTest); fclose($fhTest); }
  return (substr_count($first, ';') > substr_count($first, ',')) ? ';' : ',';
}

function tm_is_ajax_request(): bool {
  $xhr = (string)($_SERVER['HTTP_X_REQUESTED_WITH'] ?? '');
  if (strtolower($xhr) === 'xmlhttprequest') return true;
  $accept = (string)($_SERVER['HTTP_ACCEPT'] ?? '');
  if (stripos($accept, 'application/json') !== false) return true;
  return false;
}

function tm_json_out(array $payload, int $code = 200): void {
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($payload, JSON_UNESCAPED_UNICODE);
  exit;
}

/* -------------------- Core: build test report (no writes) -------------------- */
function tm_build_test_report(PDO $pdo, string $tmp, string $delim, array $map, array $expectedCanon, array $canonToDb): array {
  // IMPORTANT CHANGE: duplicates compared against ALL records, not only those visible by user.
  $existing = []; // key => list of [id, company_name, name]
  try {
    $st = $pdo->prepare("SELECT id, company_name, name FROM companies");
    $st->execute();
    while ($r = $st->fetch(PDO::FETCH_ASSOC)) {
      $k = tm_norm_name_key((string)($r['company_name'] ?? ''), (string)($r['name'] ?? ''));
      if (!isset($existing[$k])) $existing[$k] = [];
      $existing[$k][] = [
        'id' => (int)($r['id'] ?? 0),
        'company_name' => (string)($r['company_name'] ?? ''),
        'name' => (string)($r['name'] ?? ''),
      ];
    }
  } catch (Throwable $e) {
    $existing = [];
  }

  $unknownHeaders = [];
  $missingExpected = [];

  // Unknown headers are computed outside with the original header row, but keep defensive logic here:
  foreach ($expectedCanon as $need) {
    if (!isset($map[$need])) $missingExpected[] = $need;
  }

  $fh = fopen($tmp, 'r');
  if (!$fh) {
    return [
      'records' => 0,
      'missing_fields' => $missingExpected,
      'unknown_fields' => $unknownHeaders,
      'lang_invalid' => [],
      'duplicates' => [],
    ];
  }

  // Consume header row
  fgetcsv($fh, 0, $delim);

  $count = 0;
  $lineNo = 1; // header is line 1
  $langInvalidCounts = [];
  $duplicates = [];

  $langIdx = $map['language'] ?? null;
  $coIdx   = $map['company_name'] ?? null;
  $nameIdx = $map['name'] ?? null;

  while (($row = fgetcsv($fh, 0, $delim)) !== false) {
    $lineNo++;
    $allEmpty = true;
    foreach ($row as $cell) {
      if (trim((string)$cell) !== '') { $allEmpty = false; break; }
    }
    if ($allEmpty) continue;

    $count++;

    if ($langIdx !== null) {
      $lv = trim((string)($row[$langIdx] ?? ''));
      if ($lv !== '' && !preg_match('/^([a-z]{2}|[a-z]{2}-[a-z]{2})$/', $lv)) {
        $langInvalidCounts[$lv] = ($langInvalidCounts[$lv] ?? 0) + 1;
      }
    }

    if ($coIdx !== null && $nameIdx !== null) {
      $co = trim((string)($row[$coIdx] ?? ''));
      $nm = trim((string)($row[$nameIdx] ?? ''));
      if ($co !== '' && $nm !== '') {
        $k = tm_norm_name_key($co, $nm);
        if (isset($existing[$k])) {
          foreach ($existing[$k] as $ex) {
            $duplicates[] = [
              'db_id' => (int)$ex['id'],
              'db_company_name' => (string)$ex['company_name'],
              'db_name' => (string)$ex['name'],
              'csv_line' => (int)$lineNo,
              'csv_company_name' => $co,
              'csv_name' => $nm,
            ];
          }
        }
      }
    }
  }

  fclose($fh);

  return [
    'records' => $count,
    'missing_fields' => $missingExpected,
    'unknown_fields' => $unknownHeaders,
    'lang_invalid' => $langInvalidCounts,
    'duplicates' => $duplicates,
  ];
}

/* -------------------- POST handling: upload or test -------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['csv'])) {
  $do = (string)($_POST['do'] ?? 'upload'); // upload | test

  // Determine initials for UPLOAD:
  $profileInitials = tm_normalize_initials((string)($me['initials'] ?? ''));
  $importInitials = $profileInitials;

  if ($is_admin) {
    $choice = tm_normalize_initials((string)($_POST['initials_choice'] ?? ''));
    $manual = tm_normalize_initials((string)($_POST['initials_manual'] ?? ''));
    if ($manual !== '') $importInitials = $manual;
    else if ($choice !== '') $importInitials = $choice;
    else $importInitials = $profileInitials;
  }

  if ($do === 'upload') {
    if (!$is_admin) {
      if (!tm_initials_valid($importInitials)) {
        $message = "Your profile initials are missing or invalid. It must be 2 or 3 uppercase letters (A–Z). Please contact admin.";
      }
    } else {
      if (!tm_initials_valid($importInitials)) {
        $message = "Please select or enter valid initials (2–3 uppercase letters A–Z).";
      }
    }
  }

  if ($message === '' && !is_uploaded_file((string)($_FILES['csv']['tmp_name'] ?? ''))) {
    $message = "Upload error.";
  }

  $isAjax = tm_is_ajax_request();

  if ($message !== '') {
    if ($isAjax) {
      tm_json_out(['ok' => false, 'error' => $message], 400);
    }
  }

  if ($message === '') {
    $tmp = (string)$_FILES['csv']['tmp_name'];
    $delim = tm_detect_delim($tmp);

    $fh = fopen($tmp, 'r');
    if (!$fh) {
      $message = "Could not open uploaded file.";
      if ($isAjax) tm_json_out(['ok' => false, 'error' => $message], 400);
    } else {

      $header = fgetcsv($fh, 0, $delim);
      if (!$header || !is_array($header) || count($header) < 1) {
        fclose($fh);
        $message = "Invalid CSV: missing header row.";
        if ($isAjax) tm_json_out(['ok' => false, 'error' => $message], 400);
      } else {

        $map = [];              // canon => index
        $unknownHeaders = [];   // original headers not recognized

        foreach ($header as $i => $raw) {
          $rawStr = trim((string)$raw);
          if ($rawStr === '') continue;

          $canon = tm_normalize_header_key($rawStr);
          if (!isset($map[$canon])) $map[$canon] = (int)$i;

          if (!array_key_exists($canon, $canonToDb)) {
            $unknownHeaders[] = $rawStr;
          }
        }

        $unknownHeaders = array_values(array_unique(array_filter(
          $unknownHeaders,
          fn($x) => trim((string)$x) !== ''
        )));

        /* -------------------- TEST (no DB writes) -------------------- */
        if ($do === 'test') {
          // Build report reading the file from disk again (ensures we include all rows)
          fclose($fh);

          $testReport = tm_build_test_report($pdo, $tmp, $delim, $map, $expectedCanon, $canonToDb);
          // Insert unknown headers computed above
          $testReport['unknown_fields'] = $unknownHeaders;

          if ($isAjax) {
            tm_json_out([
              'ok' => true,
              'mode' => 'test',
              'beforeCount' => $beforeCount,
              'afterCount' => $beforeCount,
              'testReport' => $testReport,
            ]);
          } else {
            // Non-AJAX fallback: reopen for rendering-only path below
            $message = ''; // requirement: remove redundant "Test completed."
          }
        }

        /* -------------------- UPLOAD (DB writes) -------------------- */
        if ($do === 'upload' && $message === '') {

          $insertDbCols = [];
          $insertCanon  = [];

          foreach ($map as $canon => $idx) {
            if (isset($canonToDb[$canon])) {
              if (!$is_admin && $canon === 'private') continue;
              $insertCanon[] = $canon;
              $insertDbCols[] = $canonToDb[$canon];
            }
          }

          // Always stamp initials
          $insertCanon[] = '__initials__';
          $insertDbCols[] = 'initials';

          // Defaults if not present
          if (!in_array('private', $insertDbCols, true)) {
            $insertCanon[] = '__private_default__';
            $insertDbCols[] = 'private';
          }
          if (!in_array('status', $insertDbCols, true)) {
            $insertCanon[] = '__status_default__';
            $insertDbCols[] = 'status';
          }
          if (!in_array('wrong', $insertDbCols, true)) {
            $insertCanon[] = '__wrong_default__';
            $insertDbCols[] = 'wrong';
          }

          $colSql = implode(',', array_map(fn($c) => "`$c`", $insertDbCols));
          $phs = [];
          for ($i = 0; $i < count($insertDbCols); $i++) $phs[] = ":v{$i}";
          $sql = "INSERT INTO companies ($colSql) VALUES (" . implode(',', $phs) . ")";
          $stmt = $pdo->prepare($sql);

          // Rewind after header already read
          $pdo->beginTransaction();
          $n = 0;
          $skipped = 0;

          while (($row = fgetcsv($fh, 0, $delim)) !== false) {
            $allEmpty = true;
            foreach ($row as $cell) {
              if (trim((string)$cell) !== '') { $allEmpty = false; break; }
            }
            if ($allEmpty) { $skipped++; continue; }

            $rowCanon = [];
            foreach ($map as $canon => $idx) {
              $rowCanon[$canon] = $row[$idx] ?? '';
            }

            // linkedin autogenerate from company_name if blank
            $rawLinkedIn = trim((string)($rowCanon['linkedin'] ?? ''));
            $rawCompany  = trim((string)($rowCanon['company_name'] ?? ''));
            if ($rawLinkedIn === '' && $rawCompany !== '') {
              $generated = tm_linkedin_from_company($rawCompany);
              if ($generated !== null) $rowCanon['linkedin'] = $generated;
            }

            $bind = [];
            foreach ($insertCanon as $pos => $canonKey) {
              $ph = ":v{$pos}";

              if ($canonKey === '__initials__') { $bind[$ph] = $importInitials; continue; }
              if ($canonKey === '__private_default__') { $bind[$ph] = '0'; continue; }
              if ($canonKey === '__status_default__') { $bind[$ph] = 'Prospect'; continue; }
              if ($canonKey === '__wrong_default__') { $bind[$ph] = '0'; continue; }

              if (!$is_admin && $canonKey === 'private') {
                $bind[$ph] = '0';
                continue;
              }

              $rawVal = $rowCanon[$canonKey] ?? '';
              $bind[$ph] = tm_clean_value($canonKey, $rawVal);
            }

            $stmt->execute($bind);
            $n++;
          }

          $pdo->commit();
          fclose($fh);

          $stCnt2 = $pdo->prepare("SELECT COUNT(*) FROM companies WHERE $visClause");
          $stCnt2->execute($paramsVis);
          $afterCount = (int)$stCnt2->fetchColumn();

          $message = "Imported $n rows.";
          if ($skipped > 0) $details = "Skipped $skipped empty line(s).";

          if ($isAjax) {
            tm_json_out([
              'ok' => true,
              'mode' => 'upload',
              'beforeCount' => $beforeCount,
              'afterCount' => $afterCount,
              'message' => $message,
              'details' => $details,
            ]);
          }
        }

        if ($do === 'upload' && $message === '' && is_resource($fh)) fclose($fh);
      }
    }
  }
}

include __DIR__ . '/header.php';
?>
<section>
  <h2>Import CSV</h2>

  <form id="importForm" method="post" enctype="multipart/form-data" class="card">
    <?php if ($is_admin): ?>
      <label>Initials to use for this import - manual selection overrides (admin)</label>
      <div style="display:flex;gap:10px;flex-wrap:wrap;align-items:center;">
        <select name="initials_choice" class="search-input" style="min-width:220px;">
          <option value="">-- select from users --</option>
          <?php foreach ($userInitialsList as $ini): ?>
            <option value="<?= tm_h($ini) ?>"><?= tm_h($ini) ?></option>
          <?php endforeach; ?>
        </select>

        <input
          type="text"
          name="initials_manual"
          class="search-input"
          placeholder="Manual (AB/ABC)"
          maxlength="3"
          style="width:160px;text-transform:uppercase;"
        />
      </div>
    <?php else: ?>
      <p class="note">
        Imported records will be stamped with your initials: <strong><?= tm_h(tm_normalize_initials((string)($me['initials'] ?? ''))) ?></strong>.
      </p>
    <?php endif; ?>
<div class="stats" style="margin-top:12px;">
    <label style="margin-top:10px;">Choose CSV
      <input id="csvInput" type="file" name="csv" accept=".csv,text/csv" required>
    </label>
</div>
    <!-- Filename + buttons row (buttons hidden until file selected) -->
    <div id="fileRow" style="margin-top:10px; display:none; align-items:center; gap:10px;">
      <div
        id="fileName"
        class="note"
        style="flex:1; min-width:220px; overflow:hidden; text-overflow:ellipsis; white-space:nowrap; margin:0;"
        title=""
      ></div>

      <div id="btnRow" style="display:flex; gap:10px; flex-wrap:nowrap; align-items:center;">
        <button id="btnUpload" class="btn primary" type="button">upload</button>
        <button id="btnTest" class="btn" type="button">test</button>
      </div>
    </div>

    <noscript>
      <p class="note" style="margin-top:10px;">
        JavaScript is required for the improved import workflow. If disabled, please enable it and reload.
      </p>
    </noscript>

  </form>

  <div class="stats" style="margin-top:12px;">
    <div>Initial rows (visible): <strong id="initialRows"><?= (int)$beforeCount ?></strong></div>
    <div id="finalRowsWrap" style="display:none;">Final rows (visible): <strong id="finalRows"><?= (int)$afterCount ?></strong></div>
  </div>

  <div id="msgWrap" style="margin-top:10px; display:none;">
    <p id="msgMain" class="note" style="margin:0 0 6px 0;"></p>
    <p id="msgDetails" class="note" style="margin:0;"></p>
  </div>

  <div id="testReportWrap" class="card" style="margin-top:12px; padding:12px; display:none;">
    <h3 style="margin-top:0;">Import test report</h3>
    <pre id="testReportPre" style="white-space:pre-wrap; line-height:1.5; margin:0;"></pre>
  </div>

  <?php if ($message): ?>
    <!-- Non-AJAX fallback message (upload). Test intentionally has no redundant completion line. -->
    <p class="note" style="margin-top:10px;"><?= tm_h($message) ?></p>
  <?php endif; ?>
  <?php if ($details): ?>
    <p class="note"><?= tm_h($details) ?></p>
  <?php endif; ?>

  <?php if (is_array($testReport)): ?>
    <!-- Non-AJAX fallback report -->
    <div class="card" style="margin-top:12px; padding:12px;">
      <h3 style="margin-top:0;">Import test report</h3>
      <div style="white-space:pre-wrap; line-height:1.5;">
<?php
  echo "Nr of records to import: " . (int)$testReport['records'] . "\n\n";

  echo "Fields that are missing from the import file (optional fields not considered; private not considered):\n";
  $miss = $testReport['missing_fields'] ?? [];
  if (!$miss) echo "  (none)\n\n";
  else { foreach ($miss as $f) echo "  - {$f}\n"; echo "\n"; }

  echo "Fields that do not exist in the application:\n";
  $unk = $testReport['unknown_fields'] ?? [];
  if (!$unk) echo "  (none)\n\n";
  else { foreach ($unk as $f) echo "  - {$f}\n"; echo "\n"; }

  echo "Language field not seem correct (valid: xx OR xx-yy, lowercase):\n";
  $bad = $testReport['lang_invalid'] ?? [];
  if (!$bad) echo "  (none)\n\n";
  else { foreach ($bad as $val => $cnt) echo "  - {$val} ({$cnt})\n"; echo "\n"; }

  echo "Potential duplicates (existing DB record vs CSV line):\n";
  $dups = $testReport['duplicates'] ?? [];
  if (!$dups) {
    echo "  (none)\n";
  } else {
    foreach ($dups as $d) {
      $db = "{$d['db_id']}-{$d['db_company_name']}-{$d['db_name']}";
      $csv = "{$d['csv_line']}-{$d['csv_company_name']}-{$d['csv_name']}";
      echo "  DB: {$db}\n  CSV: {$csv}\n\n";
    }
  }
?>
      </div>
    </div>
  <?php endif; ?>

  <details style="margin-top:12px;">
    <summary>Import information rules</summary>
    <p class="note" style="margin-top:8px;">
      Expected columns (any subset accepted).
    </p>
    <p class="note" style="margin-top:8px;">
    country, language, company_name, sector, website, name, mf, role, linkedIn/linkedin, general_email, hq_city, email, grupo.
    </p>
    <p class="note" style="margin-top:8px;">
    optional: comment, history, status, wrong, last_date, last_template.
    </p>
    <p class="note" style="margin-top:8px;">
      Language rules (test): <code>en</code> or <code>en-us</code> (lowercase).
    </p>
    <p class="note" style="margin-top:10px;">
      If LinkedIn is blank, it will be generated from company_name as <code>/company/&lt;company_name&gt;/</code>.
    </p>
  </details>
</section>

<script>
(function () {
  const form = document.getElementById('importForm');
  const csvInput = document.getElementById('csvInput');

  const fileRow = document.getElementById('fileRow');
  const fileName = document.getElementById('fileName');

  const btnUpload = document.getElementById('btnUpload');
  const btnTest = document.getElementById('btnTest');

  const finalWrap = document.getElementById('finalRowsWrap');
  const finalRows = document.getElementById('finalRows');

  const msgWrap = document.getElementById('msgWrap');
  const msgMain = document.getElementById('msgMain');
  const msgDetails = document.getElementById('msgDetails');

  const testWrap = document.getElementById('testReportWrap');
  const testPre = document.getElementById('testReportPre');

  function setMessage(main, details) {
    const hasMain = !!(main && String(main).trim());
    const hasDet = !!(details && String(details).trim());
    if (!hasMain && !hasDet) {
      msgWrap.style.display = 'none';
      msgMain.textContent = '';
      msgDetails.textContent = '';
      return;
    }
    msgWrap.style.display = '';
    msgMain.textContent = hasMain ? String(main) : '';
    msgDetails.textContent = hasDet ? String(details) : '';
  }

  function buildReportText(r) {
    if (!r || typeof r !== 'object') return '';

    let out = '';
    out += 'Nr of records to import: ' + (r.records ?? 0) + '\n\n';

    out += 'Fields that are missing from the import file (optional fields not considered; private not considered):\n';
    const miss = r.missing_fields || [];
    if (!miss.length) out += '  (none)\n\n';
    else { miss.forEach(f => out += '  - ' + f + '\n'); out += '\n'; }

    out += 'Fields that do not exist in the application:\n';
    const unk = r.unknown_fields || [];
    if (!unk.length) out += '  (none)\n\n';
    else { unk.forEach(f => out += '  - ' + f + '\n'); out += '\n'; }

    out += 'Language field not seem correct (valid: xx OR xx-yy, lowercase):\n';
    const bad = r.lang_invalid || {};
    const badKeys = Object.keys(bad);
    if (!badKeys.length) out += '  (none)\n\n';
    else { badKeys.forEach(k => out += '  - ' + k + ' (' + bad[k] + ')\n'); out += '\n'; }

    out += 'Potential duplicates (existing DB record vs CSV line):\n';
    const dups = r.duplicates || [];
    if (!dups.length) out += '  (none)\n';
    else {
      dups.forEach(d => {
        const db = (d.db_id ?? '') + '-' + (d.db_company_name ?? '') + '-' + (d.db_name ?? '');
        const csv = (d.csv_line ?? '') + '-' + (d.csv_company_name ?? '') + '-' + (d.csv_name ?? '');
        out += '  DB: ' + db + '\n  CSV: ' + csv + '\n\n';
      });
    }

    return out;
  }

  function showFileUI(file) {
    if (!file) {
      fileRow.style.display = 'none';
      fileName.textContent = '';
      fileName.title = '';
      return;
    }
    fileRow.style.display = 'flex';
    fileName.textContent = file.name;
    fileName.title = file.name;

    // On new selection: show both buttons again and hide previous outputs
    btnUpload.style.display = '';
    btnUpload.disabled = false;
    btnUpload.textContent = 'upload';

    btnTest.style.display = '';
    btnTest.disabled = false;
    btnTest.textContent = 'test';

    finalWrap.style.display = 'none';
    setMessage('', '');
    testWrap.style.display = 'none';
    testPre.textContent = '';
  }

  csvInput.addEventListener('change', function () {
    showFileUI(csvInput.files && csvInput.files[0] ? csvInput.files[0] : null);
  });

  async function doAjax(mode) {
    const file = csvInput.files && csvInput.files[0] ? csvInput.files[0] : null;
    if (!file) return;

    setMessage('', '');

    const fd = new FormData(form);
    fd.set('do', mode);

    const headers = { 'X-Requested-With': 'XMLHttpRequest', 'Accept': 'application/json' };

    const res = await fetch('import.php', { method: 'POST', body: fd, headers });
    const data = await res.json().catch(() => null);

    if (!res.ok || !data || data.ok !== true) {
      const err = (data && (data.error || data.detail)) ? (data.error || data.detail) : 'Operation failed';
      throw new Error(err);
    }

    return data;
  }

  btnTest.addEventListener('click', async function () {
    if (btnTest.disabled) return;

    // requirement: hide test button after clicking; keep filename visible
    btnTest.style.display = 'none';

    // keep upload available (but disable during request)
    btnUpload.disabled = true;

    try {
      btnUpload.textContent = 'upload';
      const data = await doAjax('test');

      // requirement: remove "test completed" line -> only show the report
      const reportText = buildReportText(data.testReport || {});
      testPre.textContent = reportText;
      testWrap.style.display = '';

      // re-enable upload
      btnUpload.disabled = false;
    } catch (e) {
      // restore test button so user can retry
      btnTest.style.display = '';
      btnUpload.disabled = false;
      setMessage(String(e && e.message ? e.message : 'Test failed'), '');
    }
  });

  btnUpload.addEventListener('click', async function () {
    if (btnUpload.disabled) return;

    // requirement: hide upload button after clicking; keep filename visible
    btnUpload.style.display = 'none';

    // also keep test hidden if it was used; otherwise hide it during upload anyway
    btnTest.style.display = 'none';

    try {
      const data = await doAjax('upload');

      // show final rows info; hide test report (requirement #4)
      if (finalRows && typeof data.afterCount !== 'undefined') {
        finalRows.textContent = String(data.afterCount);
      }
      finalWrap.style.display = '';

      testWrap.style.display = 'none';
      testPre.textContent = '';

      setMessage(data.message || 'Imported.', data.details || '');

      // keep filename; allow selecting a new file to reset the UI
      // (do not clear the input automatically)
    } catch (e) {
      // restore buttons to allow retry
      btnUpload.style.display = '';
      btnUpload.disabled = false;

      // show test again only if it was not intentionally hidden by earlier test click:
      // safest approach: show it (user can still run test)
      btnTest.style.display = '';
      btnTest.disabled = false;

      setMessage(String(e && e.message ? e.message : 'Upload failed'), '');
    }
  });

  // initial state: buttons hidden, final rows hidden
  showFileUI(null);
})();
</script>

<?php include __DIR__ . '/footer.php'; ?>
