<?php
declare(strict_types=1);

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

require_login();
$me = current_user();
$role = (string)($me['role'] ?? 'user');
$is_admin = ($role === 'admin');

/**
 * Stream CSV from a query result.
 */
function stream_csv_query(PDOStatement $stmt, array $cols, string $downloadName): void {
  header('Content-Type: text/csv; charset=UTF-8');
  header('Content-Disposition: attachment; filename="'.$downloadName.'"');
  header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
  header('Pragma: no-cache');

  // UTF-8 BOM for Excel
  echo "\xEF\xBB\xBF";

  $out = fopen('php://output', 'w');
  fputcsv($out, $cols);

  while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
    $line = [];
    foreach ($cols as $c) {
      $line[] = array_key_exists($c, $row) ? $row[$c] : '';
    }
    fputcsv($out, $line);
  }
  fclose($out);
  exit;
}

/**
 * Build a safe SELECT list using explicit columns only.
 */
function select_list(array $cols): string {
  $parts = [];
  foreach ($cols as $c) {
    $parts[] = "`$c`";
  }
  return implode(',', $parts);
}

$action = (string)($_GET['action'] ?? '');
$table  = (string)($_GET['table']  ?? '');

if ($action === 'csv') {

  // Non-admin can export ONLY companies
  if (!$is_admin && $table !== 'companies') {
    http_response_code(403);
    echo "Forbidden.";
    exit;
  }

  if ($table === 'companies') {

    // Column order: align to current list order and schema
    // id field excluded from export
    $cols = [
      'company_name',
      'name',
      'mf',
      'email',
      'wrong',
      'linkedIn',
      'status',
      'language',
      'country',
      'hq_city',
      'comment',
      'history',
      'sector',
      'grupo',
      'website',
      'general_email',
      'last_date',
      'last_template',
      'initials'
    ];

    // Only admins export the private column
    if ($is_admin) {
      $cols[] = 'private';
    }

    [$visClause, $visParams] = companies_visibility_clause($me);

    // Non-admin: exclude soft-deleted records
    $softDeleteClause = $is_admin ? '' : " AND (`del` = 0 OR `del` IS NULL)";

    $sql = "SELECT " . select_list($cols) . " FROM `companies` WHERE $visClause$softDeleteClause ORDER BY id ASC";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($visParams);

    $fname = 'companies_' . date('Ymd') . '.csv';
    stream_csv_query($stmt, $cols, $fname);
  }

  if ($table === 'templates') {
    // Admin only (enforced above)
    // Include del if your table has it; if not, remove it from the list below.
    $cols = ['id','ref','title1','title2','title3','body','language','del'];
    $sql  = "SELECT " . select_list($cols) . " FROM `templates` ORDER BY id ASC";
    $stmt = $pdo->query($sql);

    $fname = 'templates_' . date('Ymd') . '.csv';
    stream_csv_query($stmt, $cols, $fname);
  }

  if ($table === 'log') {
    // Admin only (enforced above)
    $cols = ['id','id_companies','id_templates','ref','language','title_nr','timestamp'];
    $sql  = "SELECT " . select_list($cols) . " FROM `log` ORDER BY id ASC";
    $stmt = $pdo->query($sql);

    $fname = 'log_' . date('Ymd') . '.csv';
    stream_csv_query($stmt, $cols, $fname);
  }

  http_response_code(400);
  echo "Unknown table.";
  exit;
}

include __DIR__ . '/header.php';
?>
<section>
  <h2>Export CSV</h2>

  <div class="card">
    <p>Click to download a CSV snapshot.</p>

    <div class="stats">
      <a class="btn" href="export.php?action=csv&table=companies">Export companies.csv</a>

      <?php if ($is_admin): ?>
        <a class="btn" href="export.php?action=csv&table=templates">Export templates.csv</a>
        <a class="btn" href="export.php?action=csv&table=log">Export log.csv</a>
      <?php endif; ?>
    </div>

    <p class="note">
      CSV is UTF-8 with header row; fields are properly quoted.
    </p>
  </div>
</section>
<?php include __DIR__ . '/footer.php'; ?>
