<?php
declare(strict_types=1);

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

require_login();
$me = current_user();

function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function is_admin(?array $u): bool { return $u && (($u['role'] ?? '') === 'admin'); }

$admin = is_admin($me);

/* -------------------- POST: save / delete(soft) / recover -------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = (string)($_POST['action'] ?? '');
  $id     = (int)($_POST['id'] ?? 0);

  // keep filter context on POST redirects
  $redir_lang = trim((string)($_POST['redir_lang'] ?? ''));
  $redir_ref  = trim((string)($_POST['redir_ref'] ?? ''));
  $redir_show = (string)($_POST['redir_show_deleted'] ?? '');
  $qs = [];
  if ($redir_lang !== '') $qs['lang'] = $redir_lang;
  if ($redir_ref !== '')  $qs['ref']  = $redir_ref;
  if ($admin && $redir_show === '1') $qs['show_deleted'] = '1';
  $redir = 'templates.php' . (count($qs) ? ('?' . http_build_query($qs)) : '');

  if ($action === 'save') {
    $language = trim((string)($_POST['language'] ?? ''));
    $ref      = trim((string)($_POST['ref'] ?? ''));
    $title1   = trim((string)($_POST['title1'] ?? ''));
    $title2   = trim((string)($_POST['title2'] ?? ''));
    $title3   = trim((string)($_POST['title3'] ?? ''));
    $body     = trim((string)($_POST['body'] ?? ''));

    if ($id > 0) {
      $sql = "UPDATE templates
              SET language=:language, ref=:ref, title1=:title1, title2=:title2, title3=:title3, body=:body
              WHERE id=:id";
      $pdo->prepare($sql)->execute([
        ':language'=>$language, ':ref'=>$ref,
        ':title1'=>$title1, ':title2'=>$title2, ':title3'=>$title3,
        ':body'=>$body, ':id'=>$id
      ]);
    } else {
      // New templates should default to del=0
      $sql = "INSERT INTO templates (language, ref, title1, title2, title3, body, del)
              VALUES (:language,:ref,:title1,:title2,:title3,:body, 0)";
      $pdo->prepare($sql)->execute([
        ':language'=>$language, ':ref'=>$ref,
        ':title1'=>$title1, ':title2'=>$title2, ':title3'=>$title3,
        ':body'=>$body
      ]);
    }

    header('Location: ' . $redir);
    exit;
  }

  if ($action === 'delete') {
    // soft delete: del=1
    if ($id > 0) {
      $pdo->prepare("UPDATE templates SET del=1 WHERE id=:id")->execute([':id'=>$id]);
    }
    header('Location: ' . $redir);
    exit;
  }

  if ($action === 'recover') {
    if (!$admin) {
      http_response_code(403);
      echo "Forbidden";
      exit;
    }
    if ($id > 0) {
      $pdo->prepare("UPDATE templates SET del=0 WHERE id=:id")->execute([':id'=>$id]);
    }
    header('Location: ' . $redir);
    exit;
  }

  header('Location: ' . $redir);
  exit;
}

/* -------------------- GET filters -------------------- */
$selLang = trim((string)($_GET['lang'] ?? ''));
$selRef  = trim((string)($_GET['ref'] ?? ''));
$showDeleted = $admin && ((string)($_GET['show_deleted'] ?? '') === '1');

/* -------------------- Load combo options (distinct values) -------------------- */
$langs = $pdo->query("SELECT DISTINCT language FROM templates WHERE language IS NOT NULL AND TRIM(language) <> '' ORDER BY language ASC")
             ->fetchAll(PDO::FETCH_COLUMN);

$refs  = $pdo->query("SELECT DISTINCT ref FROM templates WHERE ref IS NOT NULL AND TRIM(ref) <> '' ORDER BY ref ASC")
             ->fetchAll(PDO::FETCH_COLUMN);

/* -------------------- Load templates list -------------------- */
$where = [];
$args  = [];

if ($showDeleted) {
  // Show ONLY deleted records
  $where[] = "(del = 1)";
} else {
  // Default: show only non-deleted
  $where[] = "(del IS NULL OR del = 0)";
}


if ($selLang !== '') {
  $where[] = "language = :lang";
  $args[':lang'] = $selLang;
}
if ($selRef !== '') {
  $where[] = "ref = :ref";
  $args[':ref'] = $selRef;
}

$sql = "SELECT * FROM templates";
if (count($where)) {
  $sql .= " WHERE " . implode(" AND ", $where);
}
$sql .= " ORDER BY id ASC";

$st = $pdo->prepare($sql);
$st->execute($args);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

include __DIR__ . '/header.php';
?>
<section class="tpl-page">
  <div class="toolbar" style="display:flex;align-items:center;gap:10px;flex-wrap:wrap;">
    <div class="brand" style="opacity:.75">templates</div>

    <!-- Filters (GET) -->
    <form method="get" action="templates.php" id="tplFilterForm" style="display:flex;align-items:center;gap:10px;flex-wrap:wrap;">
      <select class="search-input" name="lang" onchange="document.getElementById('tplFilterForm').submit()">
        <option value="">all languages</option>
        <?php foreach ($langs as $l): ?>
          <?php $l = (string)$l; ?>
          <option value="<?= h($l) ?>" <?= ($l === $selLang) ? 'selected' : '' ?>><?= h($l) ?></option>
        <?php endforeach; ?>
      </select>

      <select class="search-input" name="ref" onchange="document.getElementById('tplFilterForm').submit()">
        <option value="">all references</option>
        <?php foreach ($refs as $r): ?>
          <?php $r = (string)$r; ?>
          <option value="<?= h($r) ?>" <?= ($r === $selRef) ? 'selected' : '' ?>><?= h($r) ?></option>
        <?php endforeach; ?>
      </select>

      <?php if ($admin): ?>
        <button class="btn" type="submit" name="show_deleted" value="<?= $showDeleted ? '0' : '1' ?>">
          <?= $showDeleted ? 'Hide deleted' : 'Show deleted' ?>
        </button>
      <?php endif; ?>
    </form>
  </div>

<!-- New template (collapsed by default) -->
<div class="tpl-card tpl-card-new">
  <!-- Collapsed state button -->
  <div id="newTplBtnWrap" style="display:flex;justify-content:flex-end;margin:0 0 8px 0;">
    <button type="button" class="btn primary" id="newTplBtn">New template</button>
  </div>

  <!-- Expanded state label -->
  <div class="note" id="newTplLabel" style="margin:0 0 8px 0;opacity:.75;display:none;">New template</div>

  <div id="newTplPanel" style="display:none;">
    <form action="templates.php" method="post" class="tpl-form" autocomplete="off" id="newTplForm">
      <input type="hidden" name="action" value="save">
      <input type="hidden" name="redir_lang" value="<?= h($selLang) ?>">
      <input type="hidden" name="redir_ref" value="<?= h($selRef) ?>">
      <input type="hidden" name="redir_show_deleted" value="<?= $showDeleted ? '1' : '0' ?>">

      <div class="tpl-grid tpl-grid--v3">
        <div class="tpl-left">
          <div class="tpl-topline">
            <div class="cell cell-id">
              <label>ID</label>
              <input type="text" name="id" value="" placeholder="(auto)" readonly>
            </div>
            <div class="cell cell-language">
              <label>Language</label>
              <input type="text" name="language" value="" maxlength="16" placeholder="en, fr...">
            </div>
            <div class="cell cell-ref">
              <label>Ref</label>
              <input type="text" name="ref" value="" placeholder="e.g., introA">
            </div>
          </div>

          <div class="cell cell-title1 twolines">
            <label>Title 1</label>
            <textarea name="title1" rows="2"></textarea>
          </div>

          <div class="cell cell-title2 twolines">
            <label>Title 2</label>
            <textarea name="title2" rows="2"></textarea>
          </div>

          <div class="cell cell-title3 twolines">
            <label>Title 3</label>
            <textarea name="title3" rows="2"></textarea>
          </div>
        </div>

        <div class="tpl-right">
          <div class="cell cell-body">
            <label>Body</label>
            <textarea name="body" class="mono"></textarea>
          </div>
        </div>
      </div>

      <div class="tpl-actions">
        <button class="btn" type="button" id="newTplClear">clear</button>
        <button class="btn primary" type="submit" id="newTplSave">save</button>
      </div>
    </form>
  </div>
</div>

  <!-- Existing templates -->
  <?php foreach ($rows as $t): ?>
    <?php
      $tid = (int)($t['id'] ?? 0);
      $isDel = !empty($t['del']);
    ?>
    <div class="tpl-card <?= $isDel ? 'tpl-deleted' : '' ?>">
      <form action="templates.php" method="post" class="tpl-form" autocomplete="off">
        <input type="hidden" name="action" value="save">
        <input type="hidden" name="redir_lang" value="<?= h($selLang) ?>">
        <input type="hidden" name="redir_ref" value="<?= h($selRef) ?>">
        <input type="hidden" name="redir_show_deleted" value="<?= $showDeleted ? '1' : '0' ?>">

        <div class="tpl-grid tpl-grid--v3">
          <div class="tpl-left">
            <div class="tpl-topline">
              <div class="cell cell-id">
                <label>ID</label>
                <input type="text" name="id" value="<?= h($t['id'] ?? '') ?>" readonly>
              </div>
              <div class="cell cell-language">
                <label>Language</label>
                <input type="text" name="language" value="<?= h($t['language'] ?? '') ?>" maxlength="16">
              </div>
              <div class="cell cell-ref">
                <label>Ref</label>
                <input type="text" name="ref" value="<?= h($t['ref'] ?? '') ?>">
              </div>
            </div>

            <div class="cell cell-title1 twolines">
              <label>Title 1</label>
              <textarea name="title1" rows="2"><?= h($t['title1'] ?? '') ?></textarea>
            </div>

            <div class="cell cell-title2 twolines">
              <label>Title 2</label>
              <textarea name="title2" rows="2"><?= h($t['title2'] ?? '') ?></textarea>
            </div>

            <div class="cell cell-title3 twolines">
              <label>Title 3</label>
              <textarea name="title3" rows="2"><?= h($t['title3'] ?? '') ?></textarea>
            </div>
          </div>

          <div class="tpl-right">
            <div class="cell cell-body">
              <label>Body</label>
              <textarea name="body" class="mono"><?= h($t['body'] ?? '') ?></textarea>
            </div>
          </div>
        </div>

        <div class="tpl-actions">
          <?php if ($admin && $isDel): ?>
            <button class="btn" name="action" value="recover" type="submit">recover</button>
          <?php endif; ?>

          <?php if (!$isDel): ?>
            <button class="btn primary" type="submit">save</button>
            <button class="btn danger" name="action" value="delete"
                    onclick="return confirm('Delete this template? (soft delete)');"
                    type="submit">delete</button>
          <?php else: ?>
            <!-- Deleted: allow save only if you want; by default keep it read-only-ish -->
            <button class="btn" type="button" disabled>deleted</button>
          <?php endif; ?>
        </div>
      </form>
    </div>
  <?php endforeach; ?>
</section>

<script>
(function(){
  const btnWrap = document.getElementById('newTplBtnWrap');
  const btn     = document.getElementById('newTplBtn');
  const label   = document.getElementById('newTplLabel');
  const panel   = document.getElementById('newTplPanel');
  const form    = document.getElementById('newTplForm');
  const clearBtn= document.getElementById('newTplClear');

  if (!btnWrap || !btn || !label || !panel || !form || !clearBtn) return;

  function openPanel(){
    btnWrap.style.display = 'none';
    label.style.display   = 'block';
    panel.style.display   = 'block';

    // focus first editable field
    const first = form.querySelector('input[name="language"], input[name="ref"], textarea[name="title1"]');
    if (first) first.focus();
  }

  function closePanel(){
    panel.style.display   = 'none';
    label.style.display   = 'none';
    btnWrap.style.display = 'flex';
  }

  btn.addEventListener('click', openPanel);

  // Clear closes the panel after resetting
  clearBtn.addEventListener('click', function(){
    form.reset();
    closePanel();
  });

  // Save: allow submit; on success, page reloads and panel is collapsed by default.
  // (No need to closePanel() here; redirect will do it.)
})();
</script>


<?php include __DIR__ . '/footer.php'; ?>
