<?php
declare(strict_types=1);

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

if (session_status() !== PHP_SESSION_ACTIVE) {
  session_start();
}

function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }

function safe_next(string $raw): string {
  $raw = trim($raw);
  if ($raw === '') return 'list.php';
  if (preg_match('/^[a-zA-Z0-9_\-]+\.php$/', $raw)) return $raw;
  return 'list.php';
}

function detect_password_column(PDO $pdo): string {
  try {
    $cols = $pdo->query("DESCRIBE users")->fetchAll(PDO::FETCH_ASSOC);
    $names = array_map(fn($r) => (string)($r['Field'] ?? ''), $cols);
    if (in_array('password_hash', $names, true)) return 'password_hash';
    if (in_array('password', $names, true)) return 'password';
  } catch (Throwable $e) {
    error_log("login.php: detect_password_column failed: " . $e->getMessage());
  }
  return 'password_hash';
}

function users_count(PDO $pdo): int {
  try {
    return (int)$pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
  } catch (Throwable $e) {
    error_log("login.php: users_count failed: " . $e->getMessage());
    return 0;
  }
}

function find_user_by_email(PDO $pdo, string $email): ?array {
  $st = $pdo->prepare("SELECT * FROM users WHERE email = :e LIMIT 1");
  $st->execute([':e' => normalize_email($email)]);
  $u = $st->fetch(PDO::FETCH_ASSOC);
  return $u ?: null;
}

function set_user_password(PDO $pdo, int $id, string $pwdcol, string $plain): void {
  $hash = password_hash($plain, PASSWORD_DEFAULT);
  $st = $pdo->prepare("UPDATE users SET `$pwdcol` = :p WHERE id = :id");
  $st->execute([':p' => $hash, ':id' => $id]);
}

function create_first_admin(PDO $pdo, string $pwdcol, string $name, string $initials, string $email, string $password): int {
  $hash = password_hash($password, PASSWORD_DEFAULT);
  $st = $pdo->prepare("
    INSERT INTO users (name, initials, email, role, `$pwdcol`)
    VALUES (:n, :i, :e, 'admin', :p)
  ");
  $st->execute([
    ':n' => $name,
    ':i' => $initials,
    ':e' => normalize_email($email),
    ':p' => $hash
  ]);
  return (int)$pdo->lastInsertId();
}

function users_has_history_column(PDO $pdo): bool {
  try {
    $cols = $pdo->query("DESCRIBE users")->fetchAll(PDO::FETCH_ASSOC);
    foreach ($cols as $c) {
      if ((string)($c['Field'] ?? '') === 'history') return true;
    }
  } catch (Throwable $e) {
    error_log("login.php: DESCRIBE users failed: " . $e->getMessage());
  }
  return false;
}

/**
 * Enhancement #15:
 * Prepend timestamp to users.history, newest first, CR/LF separators, keep last 20.
 */
function record_login_history(PDO $pdo, int $userId): void {
  if ($userId <= 0) return;

  if (!users_has_history_column($pdo)) {
    error_log("login.php: users.history column not found; skipping history update");
    return;
  }

  $stamp = (new DateTimeImmutable('now'))->format('Y-m-d H:i:s');

  // Fetch current history (explicit assoc mode)
  $st = $pdo->prepare("SELECT `history` FROM users WHERE id = :id LIMIT 1");
  $st->execute([':id' => $userId]);
  $cur = $st->fetchColumn();
  $cur = is_string($cur) ? $cur : '';

  // Normalize line endings
  $curNorm = str_replace(["\r\n", "\r"], "\n", $cur);
  $lines = array_map('trim', explode("\n", $curNorm));
  $lines = array_values(array_filter($lines, fn($x) => $x !== ''));

  array_unshift($lines, $stamp);

  if (count($lines) > 20) {
    $lines = array_slice($lines, 0, 20);
  }

  $newHist = implode("\r\n", $lines);

  $up = $pdo->prepare("UPDATE users SET `history` = :h WHERE id = :id");
  $up->execute([':h' => $newHist, ':id' => $userId]);

  // Optional confirmation in server logs
  error_log("login.php: updated users.history for user_id={$userId} (entries=" . count($lines) . ")");
}

function login_and_redirect(PDO $pdo, int $userId, string $next): void {
  $_SESSION['topmail_user_id'] = $userId;

  // IMPORTANT: do not swallow errors silently anymore; log them.
  try {
    record_login_history($pdo, $userId);
  } catch (Throwable $e) {
    error_log("login.php: record_login_history failed: " . $e->getMessage());
  }

  header('Location: ' . ($next !== '' ? $next : 'list.php'));
  exit;
}

// Already logged in
if (is_logged_in()) {
  header('Location: list.php');
  exit;
}

$pwdcol = detect_password_column($pdo);
$hasUsers = (users_count($pdo) > 0);

$next = safe_next((string)($_GET['next'] ?? $_POST['next'] ?? 'list.php'));
$mode = $hasUsers ? 'login' : 'bootstrap';
$error = '';
$prefillEmail = trim((string)($_POST['email'] ?? $_GET['email'] ?? ''));

// GET: decide if setpwd should show
if ($hasUsers && $_SERVER['REQUEST_METHOD'] !== 'POST' && $prefillEmail !== '') {
  $u = find_user_by_email($pdo, $prefillEmail);
  if ($u) {
    $stored = (string)($u[$pwdcol] ?? '');
    if ($stored === '' || $stored === null) $mode = 'setpwd';
  }
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {

  if (!$hasUsers) {
    // Bootstrap first admin
    $name = trim((string)($_POST['name'] ?? ''));
    $initials = normalize_initials((string)($_POST['initials'] ?? ''));
    $email = normalize_email((string)($_POST['email'] ?? ''));
    $p1 = (string)($_POST['password'] ?? '');
    $p2 = (string)($_POST['password2'] ?? '');

    if ($name === '' || $initials === '' || $email === '' || $p1 === '' || $p2 === '') {
      $error = 'All fields are required.';
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
      $error = 'Please enter a valid email.';
    } elseif (!initials_valid($initials)) {
      $error = 'Initials must be 2–3 uppercase letters.';
    } elseif ($p1 !== $p2) {
      $error = 'Passwords do not match.';
    } else {
      $st = $pdo->prepare("SELECT COUNT(*) FROM users WHERE email = :e");
      $st->execute([':e' => $email]);
      if ((int)$st->fetchColumn() > 0) {
        $error = 'That email already exists.';
      } else {
        $st = $pdo->prepare("SELECT COUNT(*) FROM users WHERE initials = :i");
        $st->execute([':i' => $initials]);
        if ((int)$st->fetchColumn() > 0) {
          $error = 'Those initials already exist.';
        } else {
          $id = create_first_admin($pdo, $pwdcol, $name, $initials, $email, $p1);
          login_and_redirect($pdo, $id, $next);
        }
      }
    }

    $mode = 'bootstrap';
    $prefillEmail = $email;

  } else {
    // Existing users: login or set password
    $email = normalize_email((string)($_POST['email'] ?? ''));
    $p1 = (string)($_POST['password'] ?? '');
    $p2 = (string)($_POST['password2'] ?? '');

    if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
      $error = 'Please enter a valid email.';
      $mode = 'login';
    } else {
      $u = find_user_by_email($pdo, $email);
      if (!$u) {
        $error = 'User not found. Please contact an admin to create your account.';
        $mode = 'login';
      } else {
        $stored = (string)($u[$pwdcol] ?? '');

        if ($stored === '' || $stored === null) {
          // First login / reset: set password
          $mode = 'setpwd';
          if ($p1 === '' || $p2 === '') {
            $error = 'Please enter and confirm a new password.';
          } elseif ($p1 !== $p2) {
            $error = 'Passwords do not match.';
          } else {
            set_user_password($pdo, (int)$u['id'], $pwdcol, $p1);
            login_and_redirect($pdo, (int)$u['id'], $next);
          }
        } else {
          // Normal login
          $mode = 'login';
          if ($p1 === '') {
            $error = 'Please enter your password.';
          } else {
            if (password_verify($p1, $stored)) {
              login_and_redirect($pdo, (int)$u['id'], $next);
            } else {
              $error = 'Invalid password.';
            }
          }
        }
      }
    }

    $prefillEmail = $email;
  }
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title>Visitor mail by Clebex</title>
  <link rel="icon" href="assets/favicon.ico">
  <link rel="stylesheet" href="includes/ceo.css?v=15">
  <style>
    .login-page{min-height:100vh;display:flex;align-items:center;justify-content:center;padding:26px 14px;}
    .login-card{width:min(520px,96vw);background:var(--panel);border:1px solid var(--border);border-radius:18px;box-shadow:var(--shadow);padding:18px 18px 16px;}
    .login-head{display:flex;align-items:center;gap:12px;margin-bottom:12px;}
    .login-head img{width:38px;height:38px;border-radius:10px;object-fit:cover;}
    .login-title{font-size:18px;font-weight:800;margin:0;line-height:1.2;}
    .login-sub{margin:2px 0 0 0;opacity:.8;font-size:13px;}
    .login-form{display:grid;gap:10px;margin-top:10px;}
    .login-form label{font-size:12px;opacity:.85;display:block;margin-bottom:6px;}
    .login-form input{width:100%;}
    .login-row{display:flex;gap:10px;flex-wrap:wrap;}
    .login-row>div{flex:1;min-width:200px;}
    .login-actions{display:flex;gap:10px;justify-content:flex-end;margin-top:6px;}
    .alert{border:1px solid var(--border);border-radius:12px;padding:10px 12px;background:rgba(255,255,255,.04);margin-top:10px;white-space:pre-wrap;}
    .alert.error{border-color:rgba(255,0,0,.25);}
    .small-note{margin-top:10px;opacity:.78;font-size:12px;}
  </style>
</head>
<body class="theme-black">
  <div class="login-page">
    <div class="login-card">
      <div class="login-head">
        <img src="assets/logo.png" alt="logo">
        <div style="min-width:0">
          <p class="login-title">Visitor mail by Clebex</p>
          <p class="login-sub">
            <?php if (!$hasUsers): ?>
              First-time setup — create the initial admin account.
            <?php elseif ($mode === 'setpwd'): ?>
              First login — set your password.
            <?php else: ?>
              Sign in to continue.
            <?php endif; ?>
          </p>
        </div>
        <div style="margin-left:auto">
          <button id="themeToggle" class="btn tiny theme-toggle" title="toggle background"></button>
        </div>
      </div>

      <?php if ($error): ?>
        <div class="alert error"><?= h($error) ?></div>
      <?php endif; ?>

      <?php if (!$hasUsers): ?>
        <form class="login-form" method="post" autocomplete="off">
          <input type="hidden" name="next" value="<?= h($next) ?>">
          <div>
            <label for="name">Name</label>
            <input class="search-input" id="name" name="name" type="text" value="<?= h($_POST['name'] ?? '') ?>" required>
          </div>

          <div class="login-row">
            <div>
              <label for="initials">Initials (2–3 uppercase)</label>
              <input class="search-input" id="initials" name="initials" type="text" maxlength="3" style="text-transform:uppercase" value="<?= h($_POST['initials'] ?? '') ?>" required>
            </div>
            <div>
              <label for="email">Email</label>
              <input class="search-input" id="email" name="email" type="email" value="<?= h($prefillEmail) ?>" required>
            </div>
          </div>

          <div class="login-row">
            <div>
              <label for="password">Password</label>
              <input class="search-input" id="password" name="password" type="password" required>
            </div>
            <div>
              <label for="password2">Confirm password</label>
              <input class="search-input" id="password2" name="password2" type="password" required>
            </div>
          </div>

          <div class="login-actions">
            <button class="btn primary" type="submit">Create admin & sign in</button>
          </div>
          <div class="small-note">All fields are mandatory. Email and initials must be unique.</div>
        </form>

      <?php elseif ($mode === 'setpwd'): ?>
        <form class="login-form" method="post" autocomplete="off">
          <input type="hidden" name="next" value="<?= h($next) ?>">
          <div>
            <label for="email">Email</label>
            <input class="search-input" id="email" name="email" type="email" value="<?= h($prefillEmail) ?>" required>
          </div>

          <div class="login-row">
            <div>
              <label for="password">New password</label>
              <input class="search-input" id="password" name="password" type="password" required>
            </div>
            <div>
              <label for="password2">Confirm new password</label>
              <input class="search-input" id="password2" name="password2" type="password" required>
            </div>
          </div>

          <div class="login-actions">
            <button class="btn primary" type="submit">Set password</button>
          </div>
          <div class="small-note">Your account was created by an admin. Set a password to activate it.</div>
        </form>

      <?php else: ?>
        <form class="login-form" method="post" autocomplete="off">
          <input type="hidden" name="next" value="<?= h($next) ?>">
          <div>
            <label for="email">Email</label>
            <input class="search-input" id="email" name="email" type="email" value="<?= h($prefillEmail) ?>" required>
          </div>

          <div>
            <label for="password">Password</label>
            <input class="search-input" id="password" name="password" type="password" required>
          </div>

          <div class="login-actions">
            <button class="btn primary" type="submit">Sign in</button>
          </div>
          <div class="small-note">If you cannot sign in, ask an admin to create your user or reset your password.</div>
        </form>
      <?php endif; ?>
    </div>
  </div>

  <script>
    (function(){
      const THEMES = ['black','white','teal'];
      const COLORS = { black:'#0b0d12', white:'#ffffff', teal:'rgb(36,183,175)' };

      function applyTheme(t){
        document.body.classList.remove('theme-black','theme-white','theme-teal');
        document.body.classList.add('theme-' + t);
        sessionStorage.setItem('topmail_theme', t);

        const btn = document.getElementById('themeToggle');
        if (btn) {
          const idx = THEMES.indexOf(t);
          const next = THEMES[(idx + 1) % THEMES.length];
          btn.style.background = COLORS[next];
          btn.setAttribute('data-next', next);
        }
      }

      applyTheme(sessionStorage.getItem('topmail_theme') || 'black');

      const btn = document.getElementById('themeToggle');
      if (btn) {
        btn.addEventListener('click', function(){
          const cur = sessionStorage.getItem('topmail_theme') || 'black';
          const next = THEMES[(THEMES.indexOf(cur) + 1) % THEMES.length];
          applyTheme(next);
        });
      }
    })();
  </script>
</body>
</html>
