<?php
declare(strict_types=1);

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

require_login();
$me = current_user();

/* ---------- helpers ---------- */
if (!function_exists('json_out')) {
  function json_out($data, int $code = 200): void {
    if (!headers_sent()) {
      http_response_code($code);
      header('Content-Type: application/json; charset=utf-8');
      header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
    }
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
  }
}
if (!function_exists('h')) {
  function h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}

/*
  IMPORTANT:
  - Do NOT redeclare sanitize_sort() because includes/db.php may already define it.
  - We will call sanitize_sort() from db.php if present; otherwise, fallback locally.
*/
if (!function_exists('sanitize_sort')) {
  function sanitize_sort(string $field, array $allowed): string {
    return in_array($field, $allowed, true) ? $field : 'id';
  }
}

$role = (string)($me['role'] ?? 'user');
$is_admin   = ($role === 'admin');
$is_manager = ($role === 'manager');

/* ---------- Column visibility by role ----------
   - user: hide initials, private
   - manager: show initials, hide private
   - admin: show all
*/
$show_initials = ($is_admin || $is_manager);
$show_private  = $is_admin;

/*
  Canonical field order (Enhancement #20) + job->role change:
  role must appear immediately after status
*/
$ordered_fields = [
  'company_name','name','mf','email','wrong','linkedIn','status','role','language','country','hq_city',
  'comment','history','sector','grupo','website','general_email','last_date','last_template','initials','private'
];

/* ---------- AJAX endpoints ---------- */
if (!empty($_GET['action'])) {
  try {
    $action = (string)$_GET['action'];

    // visibility clause
    $vis = companies_visibility_clause($me);
    $visClause = $vis[0];
    $visParams = $vis[1];

    if ($action === 'countries') {
      $st = $pdo->prepare("SELECT DISTINCT IFNULL(country,'') AS v FROM companies WHERE $visClause ORDER BY v ASC");
      $st->execute($visParams);
      $rows = $st->fetchAll(PDO::FETCH_COLUMN);
      $out = [];
      foreach ($rows as $v) {
        $v = (string)$v;
        if (trim($v) === '') continue;
        $out[] = $v;
      }
      json_out(['countries' => $out]);
    }

    if ($action === 'grupos') {
      $st = $pdo->prepare("SELECT DISTINCT IFNULL(grupo,'') AS v FROM companies WHERE $visClause ORDER BY v ASC");
      $st->execute($visParams);
      $rows = $st->fetchAll(PDO::FETCH_COLUMN);
      $out = [];
      foreach ($rows as $v) {
        $v = (string)$v;
        if (trim($v) === '') continue;
        $out[] = $v;
      }
      json_out(['grupos' => $out]);
    }

    if ($action === 'sectors') {
      $st = $pdo->prepare("SELECT DISTINCT IFNULL(sector,'') AS v FROM companies WHERE $visClause ORDER BY v ASC");
      $st->execute($visParams);
      $rows = $st->fetchAll(PDO::FETCH_COLUMN);
      $out = [];
      foreach ($rows as $v) {
        $v = (string)$v;
        if (trim($v) === '') continue;
        $out[] = $v;
      }
      json_out(['sectors' => $out]);
    }

    if ($action === 'list') {
      $country = (string)($_GET['country'] ?? 'all');
      $status  = (string)($_GET['status'] ?? 'all');
      $grupo   = (string)($_GET['grupo'] ?? 'all');
      $sector  = (string)($_GET['sector'] ?? 'all');
      $q       = trim((string)($_GET['q'] ?? ''));

      $sortable = array_merge(['id'], $ordered_fields);
      $sort = sanitize_sort((string)($_GET['sort'] ?? 'id'), $sortable);
      $dir  = (isset($_GET['dir']) && strtoupper((string)$_GET['dir']) === 'DESC') ? 'DESC' : 'ASC';

      $where = [ $visClause ];
      $params = $visParams;

      if ($country !== '' && $country !== 'all') {
        $where[] = "country = :c";
        $params[':c'] = $country;
      }
      if ($status !== '' && $status !== 'all') {
        $where[] = "status = :s";
        $params[':s'] = $status;
      }
      if ($grupo !== '' && $grupo !== 'all') {
        $where[] = "grupo = :g";
        $params[':g'] = $grupo;
      }
      if ($sector !== '' && $sector !== 'all') {
        $where[] = "sector = :sec";
        $params[':sec'] = $sector;
      }

      if ($q !== '') {
        $params[':q'] = "%$q%";
        $where[] = "("
          . "CAST(id AS CHAR) LIKE :q OR company_name LIKE :q OR name LIKE :q OR mf LIKE :q OR email LIKE :q OR "
          . "linkedIn LIKE :q OR status LIKE :q OR role LIKE :q OR language LIKE :q OR country LIKE :q OR hq_city LIKE :q OR "
          . "comment LIKE :q OR history LIKE :q OR sector LIKE :q OR grupo LIKE :q OR website LIKE :q OR "
          . "general_email LIKE :q OR last_template LIKE :q OR initials LIKE :q OR "
          . "DATE_FORMAT(last_date, '%Y-%m-%d %H:%i:%s') LIKE :q"
          . ")";
      }

      $sql = "SELECT * FROM companies WHERE " . implode(" AND ", $where) . " ORDER BY `$sort` $dir";
      $stmt = $pdo->prepare($sql);
      $stmt->execute($params);
      $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

      // Total visible (unfiltered) for the user
      $st2 = $pdo->prepare("SELECT COUNT(*) FROM companies WHERE $visClause");
      $st2->execute($visParams);
      $totalVisible = (int)$st2->fetchColumn();

      json_out([
        'ok' => true,
        'rows' => $rows,
        'filtered' => count($rows),
        'total' => $totalVisible
      ]);
    }

    if ($action === 'get' && isset($_GET['id'])) {
      $id = (int)$_GET['id'];
      $st = $pdo->prepare("SELECT * FROM companies WHERE id=:id LIMIT 1");
      $st->execute([':id'=>$id]);
      $row = $st->fetch(PDO::FETCH_ASSOC);
      if (!$row) json_out(['ok'=>false,'error'=>'not found'], 404);
      if (!can_access_company_row($row, $me)) json_out(['ok'=>false,'error'=>'forbidden'], 403);
      json_out(['ok'=>true,'row'=>$row]);
    }

    if ($action === 'update' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $id = (int)($_POST['id'] ?? 0);
      $field = (string)($_POST['field'] ?? '');
      $value = $_POST['value'] ?? null;

      if ($id <= 0) json_out(['ok'=>false,'error'=>'invalid id'], 400);

      $st = $pdo->prepare("SELECT * FROM companies WHERE id=:id LIMIT 1");
      $st->execute([':id'=>$id]);
      $row = $st->fetch(PDO::FETCH_ASSOC);
      if (!$row) json_out(['ok'=>false,'error'=>'not found'], 404);
      if (!can_access_company_row($row, $me)) json_out(['ok'=>false,'error'=>'forbidden'], 403);

      $allowed = [
        'company_name','name','mf','email','wrong','linkedIn','status','role','language','country','hq_city',
        'comment','history','sector','grupo','website','general_email','last_date','last_template'
      ];
      if ($is_admin) {
        $allowed[] = 'initials';
        $allowed[] = 'private';
      }

      if (!in_array($field, $allowed, true)) json_out(['ok'=>false,'error'=>'invalid field'], 400);

      if ($field === 'wrong' || $field === 'private') {
        $value = ((string)$value === '1') ? '1' : '0';
      }

      if ($field === 'initials') {
        if (!$is_admin) json_out(['ok'=>false,'error'=>'forbidden'], 403);
        $value = strtoupper(trim((string)$value));
      }
      if ($field === 'private' && !$is_admin) {
        json_out(['ok'=>false,'error'=>'forbidden'], 403);
      }

      $sql = "UPDATE companies SET `$field` = :v WHERE id = :id";
      $pdo->prepare($sql)->execute([':v'=>$value, ':id'=>$id]);

      json_out(['ok'=>true]);
    }

    if ($action === 'add' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      $insertable = $ordered_fields;

      // users: no initials + no private
      if (!$is_admin) {
        $tmp = [];
        foreach ($insertable as $f) {
          if ($f === 'initials') continue;
          if ($f === 'private') continue;
          $tmp[] = $f;
        }
        $insertable = $tmp;
      }

      // managers: no private
      if ($is_manager) {
        $tmp = [];
        foreach ($insertable as $f) {
          if ($f === 'private') continue;
          $tmp[] = $f;
        }
        $insertable = $tmp;
      }

      $data = [];
      foreach ($insertable as $f) {
        if (array_key_exists($f, $_POST)) $data[$f] = $_POST[$f];
      }

      $data['wrong'] = (isset($data['wrong']) && ((string)$data['wrong'] === '1')) ? '1' : '0';

      if ($is_admin && array_key_exists('private', $data)) {
        $data['private'] = ((string)($data['private'] ?? '0') === '1') ? '1' : '0';
      } else {
        $data['private'] = '0';
      }

      if ($is_admin) {
        $init = isset($data['initials']) ? strtoupper(trim((string)$data['initials'])) : '';
        $data['initials'] = ($init !== '') ? $init : strtoupper(trim((string)($me['initials'] ?? '')));
      } else {
        $data['initials'] = strtoupper(trim((string)($me['initials'] ?? '')));
      }

      if (empty($data)) json_out(['ok'=>false,'error'=>'no data'], 400);

      $cols = array_keys($data);
      $phs = [];
      foreach ($cols as $c) $phs[] = ':' . $c;

      // Use backticks for safety (role may be reserved in some DB versions)
      $sql = "INSERT INTO companies (`" . implode("`,`", $cols) . "`) VALUES (" . implode(',', $phs) . ")";
      $stmt = $pdo->prepare($sql);
      $params = [];
      foreach ($cols as $c) $params[':' . $c] = $data[$c];
      $stmt->execute($params);

      json_out(['ok'=>true, 'id'=>(int)$pdo->lastInsertId()]);
    }

    if ($action === 'delete' && $_SERVER['REQUEST_METHOD'] === 'POST') {
      if (!$is_admin) json_out(['ok'=>false,'error'=>'forbidden'], 403);
      $id = (int)($_POST['id'] ?? 0);
      if (!$id) json_out(['ok'=>false,'error'=>'missing id'], 400);
      $pdo->prepare("DELETE FROM companies WHERE id = :id")->execute([':id'=>$id]);
      json_out(['ok'=>true]);
    }

    json_out(['ok'=>false,'error'=>'unknown action'], 400);

  } catch (Throwable $e) {
    json_out(['ok'=>false, 'error'=>'server', 'detail'=>$e->getMessage()], 500);
  }
}

/* ---------- Normal page render ---------- */
include __DIR__ . '/header.php';
?>
<script>
  window.TOPMAIL_ROLE = <?= json_encode($role) ?>;
  window.TOPMAIL_IS_ADMIN = <?= $is_admin ? 'true' : 'false' ?>;
</script>

<section>
  <div class="toolbar" style="display:flex;gap:10px;align-items:center;flex-wrap:wrap;">
    <select id="countryFilter" class="search-input"></select>
    <select id="statusFilter" class="search-input"></select>
    <select id="grupoFilter" class="search-input"></select>
    <select id="sectorFilter" class="search-input"></select>

    <button id="resetFilters" class="btn ghost">all</button>

    <div class="search-wrap" style="margin-left:auto;display:flex;gap:6px;align-items:center;">
      <input id="q" class="search-input" type="text" placeholder="search all fields..." autocomplete="off">
      <button id="btnClearSearch" class="btn small ghost" title="clear">×</button>
    </div>

    <button id="btnAdd" class="btn primary">add a new record</button>
  </div>

  <div class="table-wrap">
    <table id="companiesTable">
      <thead>
        <tr>
          <?php
            $cols = ['actions','id'];
            foreach ($ordered_fields as $f) {
              if ($f === 'initials' && !$show_initials) continue;
              if ($f === 'private' && !$show_private) continue;
              $cols[] = $f;
            }
            $cols[] = 'id';

            foreach ($cols as $c) {
              $label = h($c);
              $cls = ($c === 'actions') ? '' : 'sortable';
              echo "<th data-field=\"{$label}\" class=\"{$cls}\">{$label}</th>";
            }
          ?>
        </tr>
      </thead>
      <tbody></tbody>
    </table>
  </div>
</section>

<?php include __DIR__ . '/footer.php'; ?>
