<?php
require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

require_login();
$me = current_user();
$role = (string)($me['role'] ?? '');
$is_admin = ($role === 'admin');

/**
 * IMPORTANT:
 * Do NOT include/execute status.php here.
 * status.php may set Content-Type: application/json which would make the browser
 * display this whole page as source code.
 *
 * We use the required hard-coded ordered list for KPI display.
 */
$STATUSES = [
  'Prospect',
  'Email Sent',
  'Follow-up',
  'Meeting Booked',
  'Qualified',
  'Proposal Sent',
  'Negotiation',
  'Closed won',
  'Closed lost',
  'Postponed'
];

/**
 * Visibility for KPI calculations (must match app behavior: only visible records).
 * - admin: all (including private)
 * - manager: all except private
 * - user: only own initials and not private
 */
function build_visibility_where(array $me, array &$params): string {
  $role = (string)($me['role'] ?? '');
  $ini  = (string)($me['initials'] ?? '');

  if ($role === 'admin') return "1=1";
  if ($role === 'manager') return "(COALESCE(private,0) = 0)";

  // user
  $params[':ini'] = $ini;
  return "(initials = :ini AND COALESCE(private,0) = 0)";
}

/**
 * Helper: grouped counts for a single field over visible records.
 * Returns an ordered associative array: value => count (excluding blank values).
 */
function grouped_counts(PDO $pdo, string $field, string $visWhere, array $params): array {
  // Field name is controlled by this file only (do not accept user input)
  $sql = "
    SELECT IFNULL($field,'') AS v, COUNT(*) AS n
      FROM companies
     WHERE $visWhere
     GROUP BY v
     ORDER BY n DESC, v ASC
  ";

  $st = $pdo->prepare($sql);
  $st->execute($params);

  $out = [];
  while ($r = $st->fetch(PDO::FETCH_ASSOC)) {
    $v = (string)($r['v'] ?? '');
    $n = (int)($r['n'] ?? 0);
    if (trim($v) === '') continue; // exclude blanks
    $out[$v] = $n;
  }
  return $out;
}

$params = [];
$visWhere = build_visibility_where($me, $params);

// Ensure correct content type for rendering
if (!headers_sent()) {
  header('Content-Type: text/html; charset=utf-8');
}

// Total
$stmt = $pdo->prepare("SELECT COUNT(*) FROM companies WHERE $visWhere");
$stmt->execute($params);
$total = (int)$stmt->fetchColumn();

// Status counts in required order
$statusCounts = [];
$stmtStatus = $pdo->prepare("SELECT COUNT(*) FROM companies WHERE $visWhere AND status = :st");
foreach ($STATUSES as $st) {
  $p = $params;
  $p[':st'] = $st;
  $stmtStatus->execute($p);
  $statusCounts[$st] = (int)$stmtStatus->fetchColumn();
}

// Wrong email
$stmt = $pdo->prepare("SELECT COUNT(*) FROM companies WHERE $visWhere AND COALESCE(wrong,0) = 1");
$stmt->execute($params);
$wrong = (int)$stmt->fetchColumn();

// Active last 30 days (last_date within last 30 days; ignore NULL)
$stmt = $pdo->prepare("
  SELECT COUNT(*)
    FROM companies
   WHERE $visWhere
     AND last_date IS NOT NULL
     AND last_date >= (NOW() - INTERVAL 30 DAY)
");
$stmt->execute($params);
$active30 = (int)$stmt->fetchColumn();

// Private count (only shown for admin)
$privateCount = null;
if ($is_admin) {
  $stmt = $pdo->prepare("SELECT COUNT(*) FROM companies WHERE $visWhere AND COALESCE(private,0) = 1");
  $stmt->execute($params);
  $privateCount = (int)$stmt->fetchColumn();
}

// NEW: totals by country / group / sector (visible records only)
$countryCounts = grouped_counts($pdo, 'country', $visWhere, $params);
$groupCounts   = grouped_counts($pdo, 'grupo',   $visWhere, $params);
$sectorCounts  = grouped_counts($pdo, 'sector',  $visWhere, $params);

$ts = (new DateTimeImmutable('now'))->format('Y-m-d H:i:s');

include __DIR__ . '/header.php';
?>
<section style="max-width:820px">
  <div class="toolbar" style="justify-content:space-between;align-items:center">
    <div style="display:flex;align-items:center;gap:10px;flex-wrap:wrap">
      <h2 style="margin:0">KPI</h2>
      <span class="note" style="margin:0;opacity:.8">based on your visible records</span>
    </div>
    <div style="display:flex;gap:8px;flex-wrap:wrap">
      <button id="btnCopyKpi" class="btn">copy</button>
      <button id="btnPrintKpi" class="btn primary">print</button>
    </div>
  </div>

  <div id="kpiBlock" class="card" style="padding:12px;white-space:pre-wrap;line-height:1.45">
<?php
echo "Total number of records: {$total}\n";

echo "\nTotal by status:\n";
foreach ($STATUSES as $st) {
  $n = $statusCounts[$st] ?? 0;
  echo "\t{$st}: {$n}\n";
}

echo "\nWrong email: {$wrong}\n";
/** echo "Active last 30 days: {$active30}\n"; */

echo "\nTotal by country:\n";
if (!$countryCounts) {
  echo "\t(none)\n";
} else {
  foreach ($countryCounts as $k => $n) {
    echo "\t{$k}: {$n}\n";
  }
}

echo "\nTotal by group:\n";
if (!$groupCounts) {
  echo "\t(none)\n";
} else {
  foreach ($groupCounts as $k => $n) {
    echo "\t{$k}: {$n}\n";
  }
}

echo "\nTotal by sector:\n";
if (!$sectorCounts) {
  echo "\t(none)\n";
} else {
  foreach ($sectorCounts as $k => $n) {
    echo "\t{$k}: {$n}\n";
  }
}

if ($privateCount !== null) {
  echo "\nPrivate: {$privateCount}\n";
}

echo "\nTime-stamp: {$ts}\n";
?>
  </div>
</section>

<script>
(function(){
  const block = document.getElementById('kpiBlock');
  const btnCopy = document.getElementById('btnCopyKpi');
  const btnPrint = document.getElementById('btnPrintKpi');

  function textToCopy(){
    return (block && block.textContent) ? (block.textContent.trim() + "\n") : "";
  }

  if (btnCopy) {
    btnCopy.addEventListener('click', async () => {
      const txt = textToCopy();
      if (!txt) return;
      try {
        await navigator.clipboard.writeText(txt);
      } catch (e) {
        const ta = document.createElement('textarea');
        ta.value = txt;
        ta.style.position = 'fixed';
        ta.style.left = '-9999px';
        document.body.appendChild(ta);
        ta.select();
        try { document.execCommand('copy'); } catch(_) {}
        ta.remove();
      }
    });
  }

  if (btnPrint) {
    btnPrint.addEventListener('click', () => window.print());
  }
})();
</script>

<?php include __DIR__ . '/footer.php'; ?>
