<?php
// install.php — TopMail installer / schema initializer (collision-safe)
//
// Purpose:
//  - Connects to DB server, creates the database (if missing)
//  - Creates/updates required tables: companies, templates, users, log
//  - Adds missing columns/indexes (best-effort migration)
//
// Usage:
//  1) Upload this file alongside your other PHP files.
//  2) Visit: /install.php (optionally add ?debug=1)
//  3) Delete install.php afterwards (recommended)

declare(strict_types=1);

$DEBUG = isset($_GET['debug']);

// -----------------------------------------------------------------------------
// IMPORTANT: set credentials here (do NOT include includes/db.php to avoid clashes)
// -----------------------------------------------------------------------------
$DB_HOST = 'localhost'; // change if needed
$DB_PORT = 3306;
$DB_NAME = 'visitormail_campaign';
$DB_USER = 'visitormail_dbf_admin';
$DB_PASS = 'Go2026&&clebex!!';

// -----------------------------------------------------------------------------
// Helpers (prefixed to avoid collisions)
// -----------------------------------------------------------------------------
function tm_install_h(string $s): string { return htmlspecialchars($s, ENT_QUOTES, 'UTF-8'); }

function tm_install_out(string $msg, bool $ok = true): void {
  $color = $ok ? '#1f7d58' : '#b33';
  echo "<div style=\"padding:10px;margin:8px 0;border:1px solid {$color};border-radius:10px;white-space:pre-wrap;\">"
     . tm_install_h($msg)
     . "</div>";
}

function tm_install_fail(string $msg, ?Throwable $e, bool $debug): void {
  tm_install_out($msg . ($debug && $e ? ("\n\n" . $e->getMessage()) : ''), false);
  exit;
}

function tm_install_pdo_server(string $host, int $port, string $user, string $pass): PDO {
  $dsn = "mysql:host={$host};port={$port};charset=utf8mb4";
  return new PDO($dsn, $user, $pass, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
}

function tm_install_pdo_db(string $host, int $port, string $db, string $user, string $pass): PDO {
  $dsn = "mysql:host={$host};port={$port};dbname={$db};charset=utf8mb4";
  return new PDO($dsn, $user, $pass, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
}

function tm_install_db_exists(PDO $pdo, string $dbName): bool {
  $st = $pdo->prepare("SELECT SCHEMA_NAME FROM INFORMATION_SCHEMA.SCHEMATA WHERE SCHEMA_NAME = :db");
  $st->execute([':db' => $dbName]);
  return (bool)$st->fetchColumn();
}

function tm_install_ensure_db(PDO $pdoServer, string $dbName): void {
  // Best-effort: create DB if missing. If user lacks privilege, it will error.
  $pdoServer->exec("CREATE DATABASE IF NOT EXISTS `{$dbName}` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
}

function tm_install_table_exists(PDO $pdo, string $table): bool {
  $st = $pdo->prepare("SELECT 1 FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t LIMIT 1");
  $st->execute([':t'=>$table]);
  return (bool)$st->fetchColumn();
}

function tm_install_column_exists(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SELECT 1 FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t AND COLUMN_NAME = :c LIMIT 1");
  $st->execute([':t'=>$table, ':c'=>$col]);
  return (bool)$st->fetchColumn();
}

function tm_install_ensure_column(PDO $pdo, string $table, string $col, string $definition): void {
  if (!tm_install_column_exists($pdo, $table, $col)) {
    $pdo->exec("ALTER TABLE `{$table}` ADD COLUMN {$definition}");
  }
}

/**
 * If old column exists and new column does not, rename old->new with definition.
 */
function tm_install_rename_column_if_needed(PDO $pdo, string $table, string $oldCol, string $newCol, string $newDefinition): void {
  $oldExists = tm_install_column_exists($pdo, $table, $oldCol);
  $newExists = tm_install_column_exists($pdo, $table, $newCol);
  if ($oldExists && !$newExists) {
    $pdo->exec("ALTER TABLE `{$table}` CHANGE COLUMN `{$oldCol}` `{$newCol}` {$newDefinition}");
  }
}

function tm_install_ensure_index(PDO $pdo, string $table, string $indexName, string $sqlCreate): void {
  $st = $pdo->prepare("SELECT 1 FROM INFORMATION_SCHEMA.STATISTICS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = :t AND INDEX_NAME = :i LIMIT 1");
  $st->execute([':t'=>$table, ':i'=>$indexName]);
  $exists = (bool)$st->fetchColumn();
  if (!$exists) {
    $pdo->exec($sqlCreate);
  }
}

// -----------------------------------------------------------------------------
// HTML header
// -----------------------------------------------------------------------------
echo "<!doctype html><html><head><meta charset=\"utf-8\"><title>TopMail Installer</title></head><body style=\"font-family:system-ui,Segoe UI,Arial;padding:18px;max-width:980px;\">";

tm_install_out("Installer started. Target DB: {$DB_NAME}");

// -----------------------------------------------------------------------------
// Connect and ensure DB
// -----------------------------------------------------------------------------
try {
  $pdoServer = tm_install_pdo_server($DB_HOST, $DB_PORT, $DB_USER, $DB_PASS);
} catch (Throwable $e) {
  tm_install_fail("Failed to connect to DB server. Check host/user/pass.", $e, $DEBUG);
}

try {
  // Try to create DB if missing
  if (!tm_install_db_exists($pdoServer, $DB_NAME)) {
    tm_install_out("Database does not exist; attempting to create: {$DB_NAME}");
    tm_install_ensure_db($pdoServer, $DB_NAME);
    tm_install_out("Database created/ensured: {$DB_NAME}");
  } else {
    tm_install_out("Database exists: {$DB_NAME}");
  }
} catch (Throwable $e) {
  // You mentioned you may not have privileges to create DB; this will show that clearly.
  tm_install_fail("Failed to create/ensure database: {$DB_NAME}. If DB already exists and user cannot CREATE DATABASE, this can be ignored by manually creating DB.", $e, $DEBUG);
}

try {
  $pdo = tm_install_pdo_db($DB_HOST, $DB_PORT, $DB_NAME, $DB_USER, $DB_PASS);
} catch (Throwable $e) {
  tm_install_fail("Failed to connect to database {$DB_NAME}. Check credentials and DB permissions.", $e, $DEBUG);
}

// -----------------------------------------------------------------------------
// Create tables
// -----------------------------------------------------------------------------
try {
  // companies
  if (!tm_install_table_exists($pdo, 'companies')) {
    $pdo->exec("
      CREATE TABLE `companies` (
        id INT AUTO_INCREMENT PRIMARY KEY,
        country TEXT,
        language TEXT,
        company_name TEXT,
        sector TEXT,
        website TEXT,
        name TEXT,
        mf CHAR(1) NULL,
        role TEXT,
        linkedIn TEXT,
        general_email TEXT,
        hq_city TEXT,
        email TEXT,
        comment TEXT,
        history TEXT,
        status TEXT,
        wrong TINYINT(1) DEFAULT 0,
        grupo TEXT,
        last_date DATETIME NULL,
        last_template TEXT,
        initials VARCHAR(3) NULL,
        private TINYINT(1) DEFAULT 0
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");
    tm_install_out("Created table: companies");
  } else {
    tm_install_out("Table exists: companies");
  }

  // templates
  if (!tm_install_table_exists($pdo, 'templates')) {
    $pdo->exec("
      CREATE TABLE `templates` (
        id INT AUTO_INCREMENT PRIMARY KEY,
        language VARCHAR(10) NULL,
        ref VARCHAR(64) NULL,
        title1 TEXT,
        title2 TEXT,
        title3 TEXT,
        body MEDIUMTEXT,
        del TINYINT(1) DEFAULT 0
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");
    tm_install_out("Created table: templates");
  } else {
    tm_install_out("Table exists: templates");
  }

  // users
  if (!tm_install_table_exists($pdo, 'users')) {
    $pdo->exec("
      CREATE TABLE `users` (
        id INT AUTO_INCREMENT PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        initials VARCHAR(3) NOT NULL,
        email VARCHAR(255) NOT NULL UNIQUE,
        role ENUM('admin','manager','user') NOT NULL DEFAULT 'user',
        password_hash VARCHAR(255) NULL,
        history TEXT NULL
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");
    tm_install_out("Created table: users");
  } else {
    tm_install_out("Table exists: users");
  }

  // log
  if (!tm_install_table_exists($pdo, 'log')) {
    $pdo->exec("
      CREATE TABLE `log` (
        id INT AUTO_INCREMENT PRIMARY KEY,
        created_at DATETIME NOT NULL,
        user_email VARCHAR(255) NULL,
        action VARCHAR(64) NOT NULL,
        details MEDIUMTEXT NULL
      ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ");
    tm_install_out("Created table: log");
  } else {
    tm_install_out("Table exists: log");
  }

} catch (Throwable $e) {
  tm_install_fail("Failed creating tables.", $e, $DEBUG);
}

// -----------------------------------------------------------------------------
// Ensure/migrate columns and indexes (best-effort)
// -----------------------------------------------------------------------------
try {
  // companies: migrate job -> role if needed, and ensure role exists
  tm_install_rename_column_if_needed($pdo, 'companies', 'job', 'role', 'TEXT');
  tm_install_ensure_column($pdo, 'companies', 'role', "role TEXT");

  // Ensure core fields (kept for safety)
  tm_install_ensure_column($pdo, 'companies', 'wrong', "wrong TINYINT(1) DEFAULT 0");
  tm_install_ensure_column($pdo, 'companies', 'status', "status TEXT");
  tm_install_ensure_column($pdo, 'companies', 'grupo', "grupo TEXT");
  tm_install_ensure_column($pdo, 'companies', 'initials', "initials VARCHAR(3) NULL");
  tm_install_ensure_column($pdo, 'companies', 'private', "private TINYINT(1) DEFAULT 0");
  tm_install_ensure_column($pdo, 'companies', 'mf', "mf CHAR(1) NULL");
  tm_install_ensure_column($pdo, 'companies', 'last_date', "last_date DATETIME NULL");
  tm_install_ensure_column($pdo, 'companies', 'last_template', "last_template TEXT");
  tm_install_ensure_column($pdo, 'companies', 'history', "history TEXT");
  tm_install_ensure_column($pdo, 'companies', 'comment', "comment TEXT");
  tm_install_ensure_column($pdo, 'companies', 'linkedIn', "linkedIn TEXT");
  tm_install_ensure_column($pdo, 'companies', 'general_email', "general_email TEXT");
  tm_install_ensure_column($pdo, 'companies', 'hq_city', "hq_city TEXT");

  // users: ensure history
  tm_install_ensure_column($pdo, 'users', 'history', "history TEXT NULL");
  tm_install_ensure_column($pdo, 'users', 'password_hash', "password_hash VARCHAR(255) NULL");

  // templates: ensure soft delete
  tm_install_ensure_column($pdo, 'templates', 'del', "del TINYINT(1) DEFAULT 0");

  // log: basic index
  tm_install_ensure_index($pdo, 'log', 'idx_log_created_at', "CREATE INDEX idx_log_created_at ON log(created_at)");

  tm_install_out("Schema ensured successfully.");
} catch (Throwable $e) {
  tm_install_fail("Failed ensuring schema.", $e, $DEBUG);
}

tm_install_out("Done. You may delete install.php now.");

echo "</body></html>";
