<?php
// includes/db.php — DB connection + schema ensure (idempotent)

declare(strict_types=1);

// ---- DB credentials (switch host to 'localhost' on your Mac) ----
$DB_HOST = 'localhost'; // change to 'localhost' locally
$DB_PORT = 3306;
$DB_NAME = 'visitormail_campaign';
$DB_USER = 'visitormail_dbf_admin';
$DB_PASS = 'Go2026&&clebex!!';

// Optional debug with ?debug=1
$APP_DEBUG = isset($_GET['debug']);

function fatal_out($publicMsg, Throwable $e = null, $debug = false): void {
  http_response_code(500);
  echo $debug && $e
    ? "<pre style='white-space:pre-wrap;font:13px/1.4 monospace;background:#111;color:#eee;padding:12px;border:1px solid #333;border-radius:8px;'>"
      . htmlspecialchars($publicMsg . "\n\n" . $e->getMessage()) . "</pre>"
    : htmlspecialchars($publicMsg);
  exit;
}

$dsnDb   = "mysql:host={$DB_HOST};port={$DB_PORT};dbname={$DB_NAME};charset=utf8mb4";
$dsnNoDb = "mysql:host={$DB_HOST};port={$DB_PORT};charset=utf8mb4";

try {
  $pdo = new PDO($dsnDb, $DB_USER, $DB_PASS, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
  ]);
} catch (Throwable $e) {
  // Best effort: create DB if allowed
  try {
    $pdoTmp = new PDO($dsnNoDb, $DB_USER, $DB_PASS, [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]);
    $pdoTmp->exec("CREATE DATABASE IF NOT EXISTS `{$DB_NAME}` CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci");
    $pdo = new PDO($dsnDb, $DB_USER, $DB_PASS, [
      PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
      PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
  } catch (Throwable $e2) { fatal_out("DB connection error.", $e2, $APP_DEBUG); }
}

/* ---------- Helpers ---------- */
function json_out($data, int $code = 200): void {
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data);
  exit;
}
function sanitize_sort(string $field, array $allowed): string {
  return in_array($field, $allowed, true) ? $field : 'id';
}

if (!function_exists('ensure_column')) {
  function ensure_column(PDO $pdo, string $table, string $column, string $ddl): void {
    $st = $pdo->prepare("
      SELECT COUNT(*) FROM information_schema.COLUMNS
      WHERE TABLE_SCHEMA = DATABASE()
        AND TABLE_NAME = :t
        AND COLUMN_NAME = :c
    ");
    $st->execute([':t'=>$table, ':c'=>$column]);
    if ((int)$st->fetchColumn() === 0) {
      $pdo->exec("ALTER TABLE `$table` ADD COLUMN $ddl");
    }
  }
}
if (!function_exists('ensure_index')) {
  function ensure_index(PDO $pdo, string $table, string $indexName, string $ddlAddIndex): void {
    $st = $pdo->prepare("
      SELECT COUNT(*) FROM information_schema.STATISTICS
      WHERE TABLE_SCHEMA = DATABASE()
        AND TABLE_NAME = :t
        AND INDEX_NAME = :i
    ");
    $st->execute([':t'=>$table, ':i'=>$indexName]);
    if ((int)$st->fetchColumn() === 0) {
      $pdo->exec("ALTER TABLE `$table` ADD $ddlAddIndex");
    }
  }
}

/* ---------- Create tables (idempotent) ---------- */
try {
  $pdo->exec("
    CREATE TABLE IF NOT EXISTS companies (
      id INT AUTO_INCREMENT PRIMARY KEY,
      country TEXT,
      language CHAR(5),
      company_name TEXT,
      sector TEXT,
      website TEXT,
      name TEXT,
      mf CHAR(1) NULL,              -- M / F / NULL
      role TEXT,
      linkedIn TEXT,
      general_email TEXT,
      hq_city TEXT,
      email TEXT,
      wrong TINYINT(1),
      status TEXT,
      grupo TEXT,
      comment TEXT,
      last_date DATETIME NULL,
      last_template TEXT,           -- e.g. ref-lang
      history MEDIUMTEXT,           -- lines: ref-lang-YYYY-MM-DD
      initials TEXT(3),
      private TINYINT(1)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");

  $pdo->exec("
    CREATE TABLE IF NOT EXISTS templates (
      id INT AUTO_INCREMENT PRIMARY KEY,
      ref TEXT,
      title1 TEXT,
      title2 TEXT,
      title3 TEXT,
      body MEDIUMTEXT,
      language CHAR(5),
      del TINYINT(1)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");

  $pdo->exec("
    CREATE TABLE IF NOT EXISTS users (
      id INT AUTO_INCREMENT PRIMARY KEY,
      name TEXT NULL,
      initials TEXT(3) NULL,
      email TEXT,
      role TEXT,
      password_hash TEXT NULL,
      created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");

  $pdo->exec("
    CREATE TABLE IF NOT EXISTS log (
      id INT AUTO_INCREMENT PRIMARY KEY,
      id_companies INT NOT NULL,
      id_templates INT NOT NULL,
      ref TEXT,
      language CHAR(5),
      title_nr TINYINT,
      timestamp TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
      INDEX (id_companies),
      INDEX (id_templates)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
  ");
} catch (Throwable $e) {
  fatal_out("Table creation failed.", $e, $APP_DEBUG);
}

/* ---------- Migrations / required columns ---------- */
try {
  // companies additions (safe if already exist)
  ensure_column($pdo, 'companies', 'role',      "role TEXT");
  ensure_column($pdo, 'companies', 'wrong',    "wrong TINYINT(1)");
  ensure_column($pdo, 'companies', 'status',   "status TEXT");
  ensure_column($pdo, 'companies', 'grupo',    "grupo TEXT");
  ensure_column($pdo, 'companies', 'initials', "initials TEXT(3)");
  ensure_column($pdo, 'companies', 'private',  "private TINYINT(1)");

  // users additions
  ensure_column($pdo, 'users', 'password_hash', "password_hash TEXT NULL");
  ensure_column($pdo, 'users', 'created_at',    "created_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP");

  // uniqueness (best effort; may fail if duplicates exist already)
  try { ensure_index($pdo, 'users', 'uniq_users_email', "UNIQUE KEY uniq_users_email (email(190))"); } catch (Throwable $e) {}
  try { ensure_index($pdo, 'users', 'uniq_users_initials', "UNIQUE KEY uniq_users_initials (initials(10))"); } catch (Throwable $e) {}
} catch (Throwable $e) {
  fatal_out("Schema migration failed.", $e, $APP_DEBUG);
}
