/*!
 * Visitor mail by Clebex — app.js
 * List page behavior:
 * - Derive columns from THEAD (no hard-coded schema)
 * - Editable cells for all non-id fields; role included automatically
 * - Double-click popup editor for all editable fields EXCEPT tinyint flags (wrong/private)
 * - mf field rendered as dropdown (blank/M/F)
 * - Sector smart editor: dropdown uses values from sector filter options
 * - Link button behavior:
 *    • /in/ -> open linkedin profile
 *    • /company/ -> google search "linkedin <name>"
 *    • neither -> treat like company -> google search "linkedin <name>"
 *
 * Popup editor fine-tuning:
 * - Textarea fills remaining popup area (flex layout)
 * - Colors and borders follow theme variables where available
 */

/* =====================================================================================
 *  THEME HANDLING (runs on all pages)
 * ===================================================================================== */
(function themeControl() {
  const THEMES = ['black', 'white', 'teal'];
  const THEME_COLORS = { black: '#0b0d12', white: '#ffffff', teal: 'rgb(36,183,175)' };

  function applyTheme(theme) {
    const b = document.body;
    b.classList.remove('theme-black', 'theme-white', 'theme-teal');
    b.classList.add('theme-' + theme);
    sessionStorage.setItem('topmail_theme', theme);

    const btn = document.getElementById('themeToggle');
    if (btn) {
      const idx = THEMES.indexOf(theme);
      const next = THEMES[(idx + 1) % THEMES.length];
      btn.style.background = THEME_COLORS[next];
      btn.setAttribute('data-next', next);
      btn.title = 'toggle background';
    }
  }

  applyTheme(sessionStorage.getItem('topmail_theme') || 'black');

  const themeBtn = document.getElementById('themeToggle');
  if (themeBtn) {
    themeBtn.addEventListener('click', function () {
      const cur = sessionStorage.getItem('topmail_theme') || 'black';
      const order = ['black', 'white', 'teal'];
      const next = order[(order.indexOf(cur) + 1) % order.length];
      applyTheme(next);
    });
  }
})();

/* =====================================================================================
 *  LAYOUT SIZING (topbar + compact toolbar)
 * ===================================================================================== */
(function layoutSizing() {
  function setSizeVars() {
    const topbar = document.querySelector('.topbar');
    const toolbar = document.querySelector('.toolbar');
    const topH = topbar ? topbar.offsetHeight : 52;
    const toolH = toolbar ? toolbar.offsetHeight : 28;
    document.documentElement.style.setProperty('--topbar-h', topH + 'px');
    document.documentElement.style.setProperty('--toolbar-h', toolH + 'px');
  }
  setSizeVars();
  window.addEventListener('resize', setSizeVars);
})();

/* =====================================================================================
 *  GENERIC SCROLLER GUARD (prevents horizontal back/forward swipe)
 * ===================================================================================== */
(function scrollerGuard() {
  const wrap = document.querySelector('.table-wrap');
  if (!wrap) return;
  wrap.addEventListener(
    'wheel',
    function (e) {
      const horizontal = Math.abs(e.deltaX) > Math.abs(e.deltaY);
      if (!horizontal) return;
      const atLeft = wrap.scrollLeft <= 0;
      const atRight = Math.ceil(wrap.scrollLeft + wrap.clientWidth) >= wrap.scrollWidth;
      if ((e.deltaX < 0 && atLeft) || (e.deltaX > 0 && atRight)) {
        e.preventDefault();
        e.stopPropagation();
      }
    },
    { passive: false }
  );
})();

/* =====================================================================================
 *  LIST PAGE — full interactivity
 * ===================================================================================== */
(function listPage() {
  const table = document.getElementById('companiesTable');
  if (!table) return;

  const wrap = document.querySelector('.table-wrap');
  const tbody = table.tBodies[0];

  // Filters
  const selCountry = document.getElementById('countryFilter');
  const selStatus  = document.getElementById('statusFilter');
  const selGrupo   = document.getElementById('grupoFilter');
  const selSector  = document.getElementById('sectorFilter');
  const btnReset   = document.getElementById('resetFilters');

  // Search
  const qInput = document.getElementById('q');
  const btnClearSearch = document.getElementById('btnClearSearch');

  // Add
  const btnAdd = document.getElementById('btnAdd');

  const urlParams = new URLSearchParams(location.search);
  const highlightId = urlParams.get('highlight_id');

  const IS_ADMIN = !!window.TOPMAIL_IS_ADMIN;

  const TINYINT_FIELDS = new Set(['wrong', 'private']); // exclude from popup editor
  const SPECIAL_DROPDOWN_FIELDS = new Set(['mf']);      // rendered as dropdown in-table

  /* -------------------- Count UI: shown / total -------------------- */
  let shownCountEl = null, totalCountEl = null;
  (function setupCountUI() {
    const toolbar = document.querySelector('.toolbar');
    if (!toolbar) return;
    const searchWrap = toolbar.querySelector('.search-wrap');
    const holder = document.createElement('div');
    holder.className = 'count-wrap';
    holder.style.marginLeft = '8px';
    holder.style.opacity = '0.85';
    holder.innerHTML = '<span id="shownCount">0</span> / <span id="totalCount">0</span>';
    if (searchWrap) searchWrap.insertAdjacentElement('afterend', holder);
    else toolbar.appendChild(holder);
    shownCountEl = holder.querySelector('#shownCount');
    totalCountEl = holder.querySelector('#totalCount');
  })();

  /* -------------------- Column model derived from THEAD -------------------- */
  function getHeaderFields() {
    const ths = Array.from(table.querySelectorAll('thead th[data-field]'));
    return ths.map(th => (th.getAttribute('data-field') || '').trim()).filter(Boolean);
  }

  let COLS = getHeaderFields();

  function buildEditableSet() {
    const editable = new Set();
    for (const c of COLS) {
      if (c === 'actions') continue;
      if (c === 'id') continue;
      if (!IS_ADMIN && (c === 'initials' || c === 'private')) continue;
      editable.add(c);
    }
    return editable;
  }
  let EDITABLE = buildEditableSet();

  let state = {
    sortField: 'id',
    sortDir: 'ASC',
    country: 'all',
    status: 'all',
    grupo: 'all',
    sector: 'all',
    q: '',
    rows: [],
    total: 0
  };

  function updateCount() {
    if (shownCountEl) shownCountEl.textContent = String(state.rows.length);
    if (totalCountEl) totalCountEl.textContent = String(state.total || 0);
  }

  function scrollRowIntoView(tr) {
    if (!tr || !wrap) return;
    wrap.scrollTo({ top: Math.max(tr.offsetTop - 4, 0), behavior: 'smooth' });
  }

  function dateOnly(str) {
    if (!str) return '';
    const m = String(str).match(/^(\d{4}-\d{2}-\d{2})/);
    return m ? m[1] : String(str);
  }

  function normalizeLinkedIn(raw) {
    let t = (raw || '').trim();
    if (!t) return '';
    if (/^https?:\/\/(www\.)?linkedin\.com/i.test(t)) {
      t = t.replace(/^https?:\/\/(www\.)?linkedin\.com/i, '');
      if (!t.startsWith('/')) t = '/' + t;
    }
    const q = t.indexOf('?');
    if (q !== -1) t = t.slice(0, q);
    return t;
  }

  function normalizeWebsite(raw) {
    let t = (raw || '').trim();
    if (!t) return '';
    t = t.replace(/^https?:\/\/(www\.)?/i, '');
    return t;
  }

  function getSectorOptionsFromFilter() {
    if (!selSector) return [];
    const opts = Array.from(selSector.options || []);
    return opts
      .map(o => (o.value || '').trim())
      .filter(v => v && v !== 'all');
  }

  function cssVar(name, fallback) {
    const v = getComputedStyle(document.documentElement).getPropertyValue(name);
    const t = (v || '').trim();
    return t ? t : fallback;
  }

  /* -------------------- Fetch filter lists -------------------- */
  function fetchCountries() {
    if (!selCountry) return Promise.resolve();
    return fetch('list.php?action=countries')
      .then(r => r.json())
      .then(d => {
        const opts = ['all', ...((d && d.countries) || [])];
        selCountry.innerHTML = opts.map(c => `<option value="${c}">${c === 'all' ? 'all countries' : c}</option>`).join('');
        selCountry.value = state.country;
      });
  }

  function fetchGrupos() {
    if (!selGrupo) return Promise.resolve();
    return fetch('list.php?action=grupos')
      .then(r => r.json())
      .then(d => {
        const opts = ['all', ...((d && d.grupos) || [])];
        selGrupo.innerHTML = opts.map(g => `<option value="${g}">${g === 'all' ? 'all groups' : g}</option>`).join('');
        selGrupo.value = state.grupo;
      });
  }

  function fetchSectors() {
    if (!selSector) return Promise.resolve();
    return fetch('list.php?action=sectors')
      .then(r => r.json())
      .then(d => {
        const opts = ['all', ...((d && d.sectors) || [])];
        selSector.innerHTML = opts.map(s => `<option value="${s}">${s === 'all' ? 'all sectors' : s}</option>`).join('');
        selSector.value = state.sector;
      });
  }

  function ensureStatusDefault() {
    if (!selStatus) return;
    if (selStatus.options.length === 0) {
      selStatus.innerHTML = `<option value="all">all status</option>`;
    } else {
      if (selStatus.options[0] && selStatus.options[0].value === 'all') {
        selStatus.options[0].textContent = 'all status';
      }
    }
    selStatus.value = state.status;
  }

  /* -------------------- Fetch rows -------------------- */
  function fetchRows() {
    COLS = getHeaderFields();
    EDITABLE = buildEditableSet();

    const qs = new URLSearchParams({
      action: 'list',
      sort: state.sortField,
      dir: state.sortDir,
      country: state.country,
      status: state.status,
      grupo: state.grupo,
      sector: state.sector,
      q: state.q
    });

    return fetch('list.php?' + qs.toString())
      .then(r => r.json())
      .then(d => {
        state.rows = d.rows || [];
        if (typeof d.total === 'number') state.total = d.total;
        render();
        updateCount();

        if (highlightId) {
          const tr = Array.from(tbody.querySelectorAll('tr')).find(r => {
            const idCells = r.querySelectorAll('td[data-field="id"]');
            const last = idCells[idCells.length - 1];
            return last && last.textContent.trim() === String(highlightId);
          });
          if (tr) {
            selectRow(tr);
            scrollRowIntoView(tr);
          }
        }
      });
  }

  /* -------------------- Rendering -------------------- */
  function renderMfDropdown(td, row) {
    const id = row.id;
    const cur = (row.mf ?? '').toString().trim().toUpperCase();
    const sel = document.createElement('select');
    sel.className = 'search-input';
    sel.style.width = '100%';
    sel.style.minWidth = '52px';
    sel.style.padding = '6px 8px';
    sel.style.height = '30px';

    const options = ['', 'M', 'F'];
    sel.innerHTML = options.map(v => `<option value="${v}">${v}</option>`).join('');
    sel.value = (cur === 'M' || cur === 'F') ? cur : '';

    sel.addEventListener('change', () => {
      const value = sel.value;
      fetch('list.php?action=update', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({ id: String(id), field: 'mf', value })
      }).catch(() => { /* silent */ });
    });

    td.textContent = '';
    td.appendChild(sel);
  }

  function render() {
    tbody.innerHTML = '';

    state.rows.forEach(row => {
      const tr = document.createElement('tr');
      tr.dataset.rowId = row.id;

      COLS.forEach(col => {
        if (col === 'actions') {
          const act = document.createElement('td');
          act.className = 'actions-cell';
          act.dataset.field = 'actions';

          const sendA = document.createElement('a');
          sendA.className = 'btn small';
          sendA.href = 'send.php?id=' + row.id;
          sendA.textContent = 'send';

          const linkBtn = document.createElement('button');
          linkBtn.className = 'btn small';
          linkBtn.textContent = 'link';
          linkBtn.dataset.action = 'link';

          const liVal = String(row.linkedIn ?? row.linkedin ?? '').trim();
          if (!liVal) {
            linkBtn.setAttribute('disabled', 'disabled');
            linkBtn.classList.add('ghost');
          }

          const delBtn = document.createElement('button');
          delBtn.className = 'btn small danger';
          delBtn.textContent = 'del';
          delBtn.dataset.action = 'del';

          act.appendChild(sendA);
          act.appendChild(linkBtn);
          act.appendChild(delBtn);
          tr.appendChild(act);
          return;
        }

        const td = document.createElement('td');
        td.dataset.field = col;

        let val = row[col];
        if (val === undefined && col === 'linkedIn') val = row.linkedin;
        if (val === undefined && col === 'linkedin') val = row.linkedIn;

        if (col === 'last_date' && val) val = dateOnly(val);

        if (col === 'mf' && EDITABLE.has('mf')) {
          td.dataset.id = row.id;
          td.classList.add('editable');
          renderMfDropdown(td, row);
          tr.appendChild(td);
          return;
        }

        td.textContent = (val ?? '') + '';
        if (val) td.title = String(val);

        if (EDITABLE.has(col)) {
          td.contentEditable = true;
          td.classList.add('editable');
          td.dataset.id = row.id;

          // Double-click popup editor for ALL editable fields except tinyint flags and mf (mf is dropdown)
          if (!TINYINT_FIELDS.has(col) && !SPECIAL_DROPDOWN_FIELDS.has(col)) {
            td.addEventListener('dblclick', openCellEditor);
          }
        }

        tr.appendChild(td);
      });

      tbody.appendChild(tr);
    });
  }

  /* -------------------- Row selection -------------------- */
  function clearSelection() {
    tbody.querySelectorAll('tr.row-selected').forEach(tr => tr.classList.remove('row-selected'));
  }
  function selectRow(tr) {
    clearSelection();
    tr.classList.add('row-selected');
  }
  tbody.addEventListener('click', (e) => {
    const tr = e.target.closest('tr');
    if (tr) selectRow(tr);
  });

  /* -------------------- Inline save (blur) -------------------- */
  tbody.addEventListener(
    'blur',
    function (e) {
      const td = e.target;
      if (!td.classList.contains('editable')) return;
      if (!td.isContentEditable) return;

      const id = td.dataset.id;
      const field = td.dataset.field;
      if (!id || !field) return;

      let value = td.textContent.trim();

      if (field === 'linkedIn' || field === 'linkedin') value = normalizeLinkedIn(value);
      if (field === 'website') value = normalizeWebsite(value);

      td.textContent = value;
      td.title = value;

      fetch('list.php?action=update', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({ id, field, value })
      }).catch(() => { /* silent */ });

      if (field === 'linkedIn' || field === 'linkedin') {
        const btn = td.closest('tr')?.querySelector('button[data-action="link"]');
        if (btn) {
          const hasVal = value.length > 0;
          if (hasVal) {
            btn.removeAttribute('disabled');
            btn.classList.remove('ghost');
          } else {
            btn.setAttribute('disabled', 'disabled');
            btn.classList.add('ghost');
          }
        }
      }
    },
    true
  );

  /* -------------------- Sorting -------------------- */
  table.querySelectorAll('th.sortable').forEach((th) => {
    th.addEventListener('click', () => {
      const f = th.dataset.field;
      if (!f || f === 'actions') return;

      if (state.sortField === f) state.sortDir = state.sortDir === 'ASC' ? 'DESC' : 'ASC';
      else { state.sortField = f; state.sortDir = 'ASC'; }

      fetchRows();
    });
  });

  /* -------------------- Filters: auto-apply on change -------------------- */
  function readFilters() {
    state.country = selCountry ? selCountry.value : 'all';
    state.status  = selStatus  ? selStatus.value  : 'all';
    state.grupo   = selGrupo   ? selGrupo.value   : 'all';
    state.sector  = selSector  ? selSector.value  : 'all';
  }

  if (selCountry) selCountry.addEventListener('change', () => { readFilters(); fetchRows(); });
  if (selStatus)  selStatus.addEventListener('change',  () => { readFilters(); fetchRows(); });
  if (selGrupo)   selGrupo.addEventListener('change',   () => { readFilters(); fetchRows(); });
  if (selSector)  selSector.addEventListener('change',  () => { readFilters(); fetchRows(); });

  if (btnReset) {
    btnReset.addEventListener('click', () => {
      state.country = 'all';
      state.status  = 'all';
      state.grupo   = 'all';
      state.sector  = 'all';
      if (selCountry) selCountry.value = 'all';
      if (selStatus)  selStatus.value  = 'all';
      if (selGrupo)   selGrupo.value   = 'all';
      if (selSector)  selSector.value  = 'all';
      fetchRows();
    });
  }

  /* -------------------- Search -------------------- */
  let searchTimer = null;
  function triggerSearch() {
    state.q = (qInput && qInput.value) ? qInput.value.trim() : '';
    fetchRows();
  }

  if (qInput) {
    qInput.addEventListener('input', function () {
      clearTimeout(searchTimer);
      searchTimer = setTimeout(triggerSearch, 400);
    });
    qInput.addEventListener('keydown', function (e) {
      if (e.key === 'Enter') {
        e.preventDefault();
        clearTimeout(searchTimer);
        triggerSearch();
      }
      if (e.key === 'Escape') {
        qInput.value = '';
        clearTimeout(searchTimer);
        triggerSearch();
      }
    });
  }

  if (btnClearSearch) {
    btnClearSearch.addEventListener('click', function () {
      if (qInput) qInput.value = '';
      state.q = '';
      fetchRows();
    });
  }

  /* -------------------- Add button: keep existing behavior -------------------- */
  if (btnAdd) {
    btnAdd.addEventListener('click', function () {
      if (typeof window.openAddModal === 'function') {
        window.openAddModal();
        return;
      }
      alert('Add modal is not available in this build.');
    });
  }

  /* -------------------- Row actions: LINK / DEL -------------------- */
  function openGoogleSearchForLinkedIn(name) {
    const q = 'linkedin ' + (name || '').trim();
    window.open('https://www.google.com/search?q=' + encodeURIComponent(q), '_blank', 'noopener');
  }

  function openLinkedInProfilePath(path) {
    let p = normalizeLinkedIn(path);
    if (!p) return;
    if (!p.startsWith('/')) p = '/' + p;
    window.open('https://www.linkedin.com' + p, '_blank', 'noopener');
  }

  tbody.addEventListener('click', async function (e) {
    const btn = e.target.closest('button');
    if (!btn) return;
    const action = btn.dataset.action;
    const tr = btn.closest('tr');
    if (!tr) return;

    if (action === 'link') {
      const tdLI = tr.querySelector('td[data-field="linkedIn"]') || tr.querySelector('td[data-field="linkedin"]');
      const raw = tdLI ? (tdLI.textContent.trim() || '') : '';
      const li = normalizeLinkedIn(raw);

      const nm = tr.querySelector('td[data-field="name"]')?.textContent.trim() || '';
      const co = tr.querySelector('td[data-field="company_name"]')?.textContent.trim() || '';
      const searchName = nm || co;

      if (!li) return;

      const lower = li.toLowerCase();
      if (lower.includes('/in/')) {
        openLinkedInProfilePath(li);
        return;
      }
      openGoogleSearchForLinkedIn(searchName);
      return;
    }
  });

  /* -------------------- Popup editor overlay (full-size textarea + theme colors) -------------------- */
  async function clipboardReadText() {
    if (!navigator.clipboard) throw new Error('clipboard');
    return await navigator.clipboard.readText();
  }
  async function clipboardWriteText(txt) {
    if (!navigator.clipboard) throw new Error('clipboard');
    return await navigator.clipboard.writeText(txt);
  }

  function openCellEditor(ev) {
    const td = ev.currentTarget;
    const field = td.dataset.field;
    const id = td.dataset.id || (td.closest('tr')?.dataset.rowId || '');
    const editable = td.classList.contains('editable') && td.isContentEditable;

    if (TINYINT_FIELDS.has(field)) return;
    if (SPECIAL_DROPDOWN_FIELDS.has(field)) return;

    let value = td.textContent || '';

    // Theme colors (fallbacks are conservative)
    const bgCard   = cssVar('--card', 'rgba(0,0,0,0.72)');
    const fgText   = cssVar('--text', 'inherit');
    const fgMuted  = cssVar('--muted', 'rgba(255,255,255,0.7)');
    const brd      = cssVar('--border', 'rgba(255,255,255,0.22)');
    const bgInput  = cssVar('--input', 'rgba(255,255,255,0.06)');

    const backdrop = document.createElement('div');
    backdrop.className = 'cell-editor-backdrop';
    // Make backdrop consistent with modes
    backdrop.style.position = 'fixed';
    backdrop.style.inset = '0';
    backdrop.style.zIndex = '9999';
    backdrop.style.display = 'flex';
    backdrop.style.alignItems = 'center';
    backdrop.style.justifyContent = 'center';
    backdrop.style.padding = '18px';
    backdrop.style.background = 'rgba(0,0,0,0.55)';

    const panel = document.createElement('div');
    panel.className = 'cell-editor';
    panel.style.width = 'min(980px, 96vw)';
    panel.style.height = 'min(680px, 86vh)';
    panel.style.display = 'flex';
    panel.style.flexDirection = 'column';
    panel.style.gap = '10px';
    panel.style.padding = '14px';
    panel.style.borderRadius = '18px';
    panel.style.background = bgCard;
    panel.style.color = fgText;
    panel.style.border = '1px solid ' + brd;
    panel.style.boxShadow = '0 12px 40px rgba(0,0,0,0.45)';

    const titleRow = document.createElement('div');
    titleRow.style.display = 'flex';
    titleRow.style.alignItems = 'center';
    titleRow.style.justifyContent = 'space-between';
    titleRow.style.gap = '10px';

    const title = document.createElement('div');
    title.textContent = 'Edit: ' + field;
    title.style.fontSize = '16px';
    title.style.fontWeight = '700';

    // Sector smart dropdown in title row
    let sectorSelect = null;
    if (field === 'sector') {
      const opts = getSectorOptionsFromFilter();
      sectorSelect = document.createElement('select');
      sectorSelect.className = 'search-input';
      sectorSelect.style.minWidth = '220px';
      sectorSelect.style.maxWidth = '50%';
      sectorSelect.style.borderRadius = '12px';
      sectorSelect.style.background = bgInput;
      sectorSelect.style.color = fgText;
      sectorSelect.style.border = '1px solid ' + brd;
      sectorSelect.innerHTML =
        `<option value="">select sector…</option>` +
        opts.map(v => `<option value="${v}">${v}</option>`).join('');
      titleRow.appendChild(title);
      titleRow.appendChild(sectorSelect);
    } else {
      titleRow.appendChild(title);
    }

    const meta = document.createElement('div');
    meta.textContent = 'Record ID: ' + (id || '(new)');
    meta.style.fontSize = '12px';
    meta.style.color = fgMuted;

    const btnRow = document.createElement('div');
    btnRow.style.display = 'flex';
    btnRow.style.gap = '8px';
    btnRow.style.flexWrap = 'wrap';

    const bCancel = document.createElement('button'); bCancel.className = 'btn ghost'; bCancel.textContent = 'more cancel';
    const bClear  = document.createElement('button'); bClear.className  = 'btn danger'; bClear.textContent  = 'clear';
    const bCopy   = document.createElement('button'); bCopy.className   = 'btn'; bCopy.textContent   = 'copy';
    const bPaste  = document.createElement('button'); bPaste.className  = 'btn'; bPaste.textContent  = 'paste';
    const bSave   = document.createElement('button'); bSave.className   = 'btn primary'; bSave.textContent = 'save';

    btnRow.append(bCancel, bClear, bCopy, bPaste, bSave);

    const ta = document.createElement('textarea');
    ta.value = value;
    if (!editable) ta.readOnly = true;

    // Key change: textarea uses the whole remaining popup area
    ta.style.flex = '1';
    ta.style.width = '100%';
    ta.style.height = '100%';
    ta.style.resize = 'none';
    ta.style.borderRadius = '14px';
    ta.style.padding = '12px';
    ta.style.border = '1px solid ' + brd;
    ta.style.background = bgInput;
    ta.style.color = fgText;
    ta.style.lineHeight = '1.35';
    ta.style.fontFamily = 'ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace';
    ta.style.fontSize = '13px';
    ta.style.outline = 'none';

    panel.appendChild(titleRow);
    panel.appendChild(meta);
    panel.appendChild(btnRow);
    panel.appendChild(ta);

    let onKeyDown = null;
    function close() {
      document.removeEventListener('keydown', onKeyDown, true);
      if (backdrop.parentNode) document.body.removeChild(backdrop);
    }

    async function saveValueAndClose(newVal) {
      if (!editable) { close(); return; }

      let v = (newVal ?? '').toString();

      if (field === 'linkedIn' || field === 'linkedin') v = normalizeLinkedIn(v);
      if (field === 'website') v = normalizeWebsite(v);

      td.textContent = v;
      td.title = v;

      await fetch('list.php?action=update', {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({ id, field, value: v })
      }).catch(() => { /* silent */ });

      if (field === 'linkedIn' || field === 'linkedin') {
        const linkBtn = td.closest('tr')?.querySelector('button[data-action="link"]');
        if (linkBtn) {
          if (v) { linkBtn.removeAttribute('disabled'); linkBtn.classList.remove('ghost'); }
          else { linkBtn.setAttribute('disabled', 'disabled'); linkBtn.classList.add('ghost'); }
        }
      }

      close();
    }

    bCancel.addEventListener('click', () => close());
    bClear.addEventListener('click', async () => { await saveValueAndClose(''); });

    bCopy.addEventListener('click', async () => {
      try { await clipboardWriteText(ta.value); }
      catch (_) { alert('Clipboard not available'); }
      close();
    });

    bPaste.addEventListener('click', async () => {
      if (!editable) { close(); return; }
      try {
        const txt = await clipboardReadText();
        await saveValueAndClose(txt);
      } catch (_) {
        alert('Clipboard not available');
        close();
      }
    });

    bSave.addEventListener('click', async () => { await saveValueAndClose(ta.value); });

    if (sectorSelect) {
      sectorSelect.addEventListener('change', async () => {
        const v = sectorSelect.value || '';
        if (!v) return;
        await saveValueAndClose(v);
      });
    }

    onKeyDown = function (e) {
      if (e.key === 'Escape') { e.preventDefault(); close(); }
      // Ctrl/Cmd+Enter saves
      if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
        e.preventDefault();
        saveValueAndClose(ta.value);
      }
    };
    document.addEventListener('keydown', onKeyDown, true);

    backdrop.addEventListener('click', (e) => { if (e.target === backdrop) close(); });

    backdrop.appendChild(panel);
    document.body.appendChild(backdrop);

    ta.focus();
    try { ta.select(); } catch (_) {}
  }

  /* -------------------- Init -------------------- */
  ensureStatusDefault();
  Promise.all([fetchCountries(), fetchGrupos(), fetchSectors()])
    .then(() => fetchRows());
})();
