<?php
declare(strict_types=1);

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

require_login();
$me = current_user();
if (!$me) {
  header('Location: login.php');
  exit;
}

$roleUser = (string)($me['role'] ?? 'user');
$is_admin = ($roleUser === 'admin');

/* -------------------- Safe, namespaced helpers (no redeclare) -------------------- */
if (!function_exists('tm_h')) {
  function tm_h($s): string { return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
}

if (!function_exists('tm_normalize_initials')) {
  function tm_normalize_initials(string $initials): string {
    $s = strtoupper(trim($initials));
    $s = preg_replace('/[^A-Z]/', '', $s); // keep only A-Z
    return $s ?? '';
  }
}

if (!function_exists('tm_initials_valid')) {
  function tm_initials_valid(string $initials): bool {
    return (bool)preg_match('/^[A-Z]{2,3}$/', $initials);
  }
}

if (!function_exists('tm_norm_bool01')) {
  function tm_norm_bool01($v): string {
    $t = strtoupper(trim((string)$v));
    return ($t === '1' || $t === 'Y' || $t === 'YES' || $t === 'TRUE') ? '1' : '0';
  }
}

if (!function_exists('tm_normalize_header_key')) {
  function tm_normalize_header_key(string $raw): string {
    $k = trim($raw);
    $k = preg_replace('/^\xEF\xBB\xBF/', '', $k); // strip BOM
    $k = strtolower($k);
    $k = str_replace([' ', '-', '.', '/', '\\'], '_', $k);
    $k = preg_replace('/_+/', '_', $k);

    $aliases = [
      'linkedin'      => 'linkedin',
      'linkedin_url'  => 'linkedin',
      'generalemail'  => 'general_email',
      'hqcity'        => 'hq_city',
      'lastdate'      => 'last_date',
      'lasttemplate'  => 'last_template',
      'm_f'           => 'mf',
      'm__f'          => 'mf',
      'mf'            => 'mf',
      'job'           => 'role', // legacy header mapping
    ];

    return $aliases[$k] ?? $k;
  }
}

if (!function_exists('tm_normalize_last_date')) {
  function tm_normalize_last_date($val): ?string {
    $v = trim((string)$val);
    if ($v === '') return null;

    if (preg_match('/^\d{4}-\d{2}-\d{2}(\s+\d{2}:\d{2}(:\d{2})?)?$/', $v)) {
      if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $v)) return $v . ' 00:00:00';
      if (preg_match('/^\d{4}-\d{2}-\d{2}\s+\d{2}:\d{2}$/', $v)) return $v . ':00';
      return $v;
    }

    if (preg_match('/^(\d{2})\.(\d{2})\.(\d{4})(\s+(\d{2}):(\d{2}))?$/', $v, $m)) {
      $dd = $m[1]; $mm = $m[2]; $yy = $m[3];
      $hh = $m[5] ?? '00';
      $mi = $m[6] ?? '00';
      return "{$yy}-{$mm}-{$dd} {$hh}:{$mi}:00";
    }

    return $v;
  }
}

if (!function_exists('tm_linkedin_from_company')) {
  function tm_linkedin_from_company(string $companyName): ?string {
    $companyName = trim($companyName);
    if ($companyName === '') return null;

    $s = mb_strtolower($companyName, 'UTF-8');
    $s = str_replace(["\t", "\r", "\n"], ' ', $s);
    $s = preg_replace("/[’']/u", '', $s);
    $s = preg_replace('/[^a-z0-9]+/u', ' ', $s);
    $s = preg_replace('/\s+/', ' ', trim($s));
    if ($s === '') return null;

    $slug = str_replace(' ', '-', $s);
    $slug = trim($slug, '-');
    if ($slug === '') return null;

    return "/company/{$slug}/";
  }
}

if (!function_exists('tm_clean_value')) {
  function tm_clean_value(string $canonKey, $raw) {
    $v = trim((string)$raw);

    if ($canonKey === 'mf') {
      if ($v === '') return null;
      $c = strtoupper(substr($v, 0, 1));
      return ($c === 'M' || $c === 'F') ? $c : null;
    }

    if ($canonKey === 'linkedin') {
      if ($v === '') return null;

      if (preg_match('~^https?://(www\.)?linkedin\.com~i', $v)) {
        $v = preg_replace('~^https?://(www\.)?linkedin\.com~i', '', $v);
        if ($v !== '' && $v[0] !== '/') $v = '/' . $v;
      }
      $q = strpos($v, '?');
      if ($q !== false) $v = substr($v, 0, $q);

      return ($v === '' ? null : $v);
    }

    if ($canonKey === 'website') {
      if ($v === '') return null;
      $v = preg_replace('~^https?://(www\.)?~i', '', $v);
      return ($v === '' ? null : $v);
    }

    if ($canonKey === 'wrong' || $canonKey === 'private') {
      return tm_norm_bool01($v);
    }

    if ($canonKey === 'last_date') {
      return tm_normalize_last_date($v);
    }

    return ($v === '' ? null : $v);
  }
}

/* -------------------- NEW: normalize company_name + name for duplicate test -------------------- */
if (!function_exists('tm_norm_name_key')) {
  function tm_norm_name_key(string $company, string $name): string {
    $c = trim(mb_strtolower($company, 'UTF-8'));
    $n = trim(mb_strtolower($name, 'UTF-8'));
    $c = preg_replace('/\s+/', ' ', $c) ?? $c;
    $n = preg_replace('/\s+/', ' ', $n) ?? $n;
    return $c . '||' . $n;
  }
}

/* -------------------- Configuration: application fields -------------------- */
$canonToDb = [
  'country'        => 'country',
  'language'       => 'language',
  'company_name'   => 'company_name',
  'sector'         => 'sector',
  'website'        => 'website',
  'name'           => 'name',
  'mf'             => 'mf',
  'role'           => 'role',
  'linkedin'       => 'linkedIn',
  'general_email'  => 'general_email',
  'hq_city'        => 'hq_city',
  'email'          => 'email',
  'comment'        => 'comment',
  'history'        => 'history',
  'status'         => 'status',
  'grupo'          => 'grupo',
  'wrong'          => 'wrong',
  'last_date'      => 'last_date',
  'last_template'  => 'last_template',
];
if ($is_admin) $canonToDb['private'] = 'private';

$expectedCanon = [
  'country','language','company_name','sector','website','name','mf','role','linkedin',
  'general_email','hq_city','email','grupo'
];

/* -------------------- Visibility + counts (visible rows) -------------------- */
$paramsVis = [];
$visClause = '1=1';
if (function_exists('companies_visibility_clause')) {
  [$visClause, $paramsVis] = companies_visibility_clause($me);
}

$stCnt = $pdo->prepare("SELECT COUNT(*) FROM companies WHERE $visClause");
$stCnt->execute($paramsVis);
$beforeCount = (int)$stCnt->fetchColumn();
$afterCount = $beforeCount;

/* -------------------- Admin initials list for dropdown -------------------- */
$userInitialsList = [];
if ($is_admin) {
  try {
    $rows = $pdo->query("SELECT DISTINCT initials FROM users WHERE initials IS NOT NULL AND initials<>'' ORDER BY initials ASC")
                ->fetchAll(PDO::FETCH_COLUMN);
    foreach ($rows as $ini) {
      $ini = tm_normalize_initials((string)$ini);
      if ($ini !== '' && !in_array($ini, $userInitialsList, true)) $userInitialsList[] = $ini;
    }
  } catch (Throwable $e) {
    $userInitialsList = [];
  }
}

$message = '';
$details = '';
$testReport = null;

function tm_detect_delim(string $tmpPath): string {
  $first = '';
  $fhTest = fopen($tmpPath, 'r');
  if ($fhTest) { $first = (string)fgets($fhTest); fclose($fhTest); }
  return (substr_count($first, ';') > substr_count($first, ',')) ? ';' : ',';
}

/* -------------------- POST handling: upload or test -------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['csv'])) {

  $do = (string)($_POST['do'] ?? 'upload'); // upload | test

  // Determine initials for UPLOAD:
  $profileInitials = tm_normalize_initials((string)($me['initials'] ?? ''));
  $importInitials = $profileInitials;

  if ($is_admin) {
    $choice = tm_normalize_initials((string)($_POST['initials_choice'] ?? ''));
    $manual = tm_normalize_initials((string)($_POST['initials_manual'] ?? ''));
    if ($manual !== '') $importInitials = $manual;
    else if ($choice !== '') $importInitials = $choice;
    else $importInitials = $profileInitials;
  }

  if ($do === 'upload') {
    if (!$is_admin) {
      if (!tm_initials_valid($importInitials)) {
        $message = "Your profile initials are missing or invalid. It must be 2 or 3 uppercase letters (A–Z). Please contact admin.";
      }
    } else {
      if (!tm_initials_valid($importInitials)) {
        $message = "Please select or enter valid initials (2–3 uppercase letters A–Z).";
      }
    }
  }

  if ($message === '' && !is_uploaded_file((string)($_FILES['csv']['tmp_name'] ?? ''))) {
    $message = "Upload error.";
  }

  if ($message === '') {
    $tmp = (string)$_FILES['csv']['tmp_name'];
    $delim = tm_detect_delim($tmp);

    $fh = fopen($tmp, 'r');
    if (!$fh) {
      $message = "Could not open uploaded file.";
    } else {

      $header = fgetcsv($fh, 0, $delim);
      if (!$header || !is_array($header) || count($header) < 1) {
        fclose($fh);
        $message = "Invalid CSV: missing header row.";
      } else {

        $map = [];              // canon => index
        $unknownHeaders = [];   // original headers not recognized

        foreach ($header as $i => $raw) {
          $rawStr = trim((string)$raw);
          if ($rawStr === '') continue;

          $canon = tm_normalize_header_key($rawStr);
          if (!isset($map[$canon])) $map[$canon] = (int)$i;

          if (!array_key_exists($canon, $canonToDb)) {
            $unknownHeaders[] = $rawStr;
          }
        }

        $missingExpected = [];
        foreach ($expectedCanon as $need) {
          if (!isset($map[$need])) $missingExpected[] = $need;
        }

        /* -------------------- TEST (no DB writes) -------------------- */
        if ($do === 'test') {

          // Build lookup of existing visible records by (company_name + name)
          $existing = []; // key => list of [id, company_name, name]
          try {
            $st = $pdo->prepare("SELECT id, company_name, name FROM companies WHERE $visClause");
            $st->execute($paramsVis);
            while ($r = $st->fetch(PDO::FETCH_ASSOC)) {
              $k = tm_norm_name_key((string)($r['company_name'] ?? ''), (string)($r['name'] ?? ''));
              if (!isset($existing[$k])) $existing[$k] = [];
              $existing[$k][] = [
                'id' => (int)($r['id'] ?? 0),
                'company_name' => (string)($r['company_name'] ?? ''),
                'name' => (string)($r['name'] ?? ''),
              ];
            }
          } catch (Throwable $e) {
            // If something fails, we still run the rest of the test; duplicates section will be empty.
            $existing = [];
          }

          $count = 0;
          $lineNo = 1; // header is line 1
          $langInvalidCounts = [];
          $duplicates = [];

          $langIdx = $map['language'] ?? null;
          $coIdx = $map['company_name'] ?? null;
          $nameIdx = $map['name'] ?? null;

          while (($row = fgetcsv($fh, 0, $delim)) !== false) {
            $lineNo++; // CSV physical line number
            $allEmpty = true;
            foreach ($row as $cell) {
              if (trim((string)$cell) !== '') { $allEmpty = false; break; }
            }
            if ($allEmpty) continue;

            $count++;

            // Language rule: xx OR xx-yy (lowercase)
            if ($langIdx !== null) {
              $lv = trim((string)($row[$langIdx] ?? ''));
              if ($lv !== '' && !preg_match('/^([a-z]{2}|[a-z]{2}-[a-z]{2})$/', $lv)) {
                $langInvalidCounts[$lv] = ($langInvalidCounts[$lv] ?? 0) + 1;
              }
            }

            // Duplicate rule: match existing visible records by company_name + name
            if ($coIdx !== null && $nameIdx !== null) {
              $co = trim((string)($row[$coIdx] ?? ''));
              $nm = trim((string)($row[$nameIdx] ?? ''));
              if ($co !== '' && $nm !== '') {
                $k = tm_norm_name_key($co, $nm);
                if (isset($existing[$k])) {
                  foreach ($existing[$k] as $ex) {
                    $duplicates[] = [
                      'db_id' => (int)$ex['id'],
                      'db_company_name' => (string)$ex['company_name'],
                      'db_name' => (string)$ex['name'],
                      'csv_line' => (int)$lineNo,
                      'csv_company_name' => $co,
                      'csv_name' => $nm,
                    ];
                  }
                }
              }
            }
          }

          fclose($fh);

          $unknownHeaders = array_values(array_unique(array_filter(
            $unknownHeaders,
            fn($x) => trim((string)$x) !== ''
          )));

          $testReport = [
            'records' => $count,
            'missing_fields' => $missingExpected,
            'unknown_fields' => $unknownHeaders,
            'lang_invalid' => $langInvalidCounts,
            'duplicates' => $duplicates,
          ];

          $message = "Test completed.";
        }

        /* -------------------- UPLOAD (DB writes) -------------------- */
        if ($do === 'upload' && $message === '') {

          $insertDbCols = [];
          $insertCanon  = [];

          foreach ($map as $canon => $idx) {
            if (isset($canonToDb[$canon])) {
              if (!$is_admin && $canon === 'private') continue;
              $insertCanon[] = $canon;
              $insertDbCols[] = $canonToDb[$canon];
            }
          }

          // Always stamp initials
          $insertCanon[] = '__initials__';
          $insertDbCols[] = 'initials';

          // Defaults if not present
          if (!in_array('private', $insertDbCols, true)) {
            $insertCanon[] = '__private_default__';
            $insertDbCols[] = 'private';
          }
          if (!in_array('status', $insertDbCols, true)) {
            $insertCanon[] = '__status_default__';
            $insertDbCols[] = 'status';
          }
          if (!in_array('wrong', $insertDbCols, true)) {
            $insertCanon[] = '__wrong_default__';
            $insertDbCols[] = 'wrong';
          }

          $colSql = implode(',', array_map(fn($c) => "`$c`", $insertDbCols));
          $phs = [];
          for ($i = 0; $i < count($insertDbCols); $i++) $phs[] = ":v{$i}";
          $sql = "INSERT INTO companies ($colSql) VALUES (" . implode(',', $phs) . ")";
          $stmt = $pdo->prepare($sql);

          $pdo->beginTransaction();
          $n = 0;
          $skipped = 0;

          while (($row = fgetcsv($fh, 0, $delim)) !== false) {
            $allEmpty = true;
            foreach ($row as $cell) {
              if (trim((string)$cell) !== '') { $allEmpty = false; break; }
            }
            if ($allEmpty) { $skipped++; continue; }

            $rowCanon = [];
            foreach ($map as $canon => $idx) {
              $rowCanon[$canon] = $row[$idx] ?? '';
            }

            // linkedin autogenerate from company_name if blank
            $rawLinkedIn = trim((string)($rowCanon['linkedin'] ?? ''));
            $rawCompany  = trim((string)($rowCanon['company_name'] ?? ''));
            if ($rawLinkedIn === '' && $rawCompany !== '') {
              $generated = tm_linkedin_from_company($rawCompany);
              if ($generated !== null) $rowCanon['linkedin'] = $generated;
            }

            $bind = [];
            foreach ($insertCanon as $pos => $canonKey) {
              $ph = ":v{$pos}";

              if ($canonKey === '__initials__') { $bind[$ph] = $importInitials; continue; }
              if ($canonKey === '__private_default__') { $bind[$ph] = '0'; continue; }
              if ($canonKey === '__status_default__') { $bind[$ph] = 'Prospect'; continue; }
              if ($canonKey === '__wrong_default__') { $bind[$ph] = '0'; continue; }

              if (!$is_admin && $canonKey === 'private') {
                $bind[$ph] = '0';
                continue;
              }

              $rawVal = $rowCanon[$canonKey] ?? '';
              $bind[$ph] = tm_clean_value($canonKey, $rawVal);
            }

            $stmt->execute($bind);
            $n++;
          }

          $pdo->commit();
          fclose($fh);

          $stCnt2 = $pdo->prepare("SELECT COUNT(*) FROM companies WHERE $visClause");
          $stCnt2->execute($paramsVis);
          $afterCount = (int)$stCnt2->fetchColumn();

          $message = "Imported $n rows.";
          if ($skipped > 0) $details = "Skipped $skipped empty line(s).";
        }

        if ($do === 'upload' && $message === '') fclose($fh);
      }
    }
  }
}

include __DIR__ . '/header.php';
?>
<section>
  <h2>Import CSV</h2>

  <form method="post" enctype="multipart/form-data" class="card">
    <?php if ($is_admin): ?>
      <label>Initials to use for this import (admin)</label>
      <div style="display:flex;gap:10px;flex-wrap:wrap;align-items:center;">
        <select name="initials_choice" class="search-input" style="min-width:220px;">
          <option value="">-- select from users --</option>
          <?php foreach ($userInitialsList as $ini): ?>
            <option value="<?= tm_h($ini) ?>"><?= tm_h($ini) ?></option>
          <?php endforeach; ?>
        </select>

        <input
          type="text"
          name="initials_manual"
          class="search-input"
          placeholder="Manual (AB/ABC)"
          maxlength="3"
          style="width:160px;text-transform:uppercase;"
        />
      </div>
      <p class="note" style="margin-top:8px;">
        Manual initials override the dropdown.
      </p>
    <?php else: ?>
      <p class="note">
        Imported records will be stamped with your initials: <strong><?= tm_h(tm_normalize_initials((string)($me['initials'] ?? ''))) ?></strong>.
      </p>
    <?php endif; ?>

    <label style="margin-top:10px;">Choose CSV
      <input type="file" name="csv" accept=".csv,text/csv" required>
    </label>

    <div style="margin-top:10px; display:flex; gap:10px; flex-wrap:wrap;">
      <button class="btn primary" type="submit" name="do" value="upload">upload</button>
      <button class="btn" type="submit" name="do" value="test">test</button>
    </div>

    <p class="note" style="margin-top:10px;">
      If LinkedIn is blank, it will be generated from company_name as <code>/company/&lt;company_name&gt;/</code>.
    </p>
  </form>

  <div class="stats">
    <div>Initial rows (visible): <strong><?= (int)$beforeCount ?></strong></div>
    <div>Final rows (visible): <strong><?= (int)$afterCount ?></strong></div>
  </div>

  <?php if ($message): ?><p class="note"><?= tm_h($message) ?></p><?php endif; ?>
  <?php if ($details): ?><p class="note"><?= tm_h($details) ?></p><?php endif; ?>

  <?php if (is_array($testReport)): ?>
    <div class="card" style="margin-top:12px; padding:12px;">
      <h3 style="margin-top:0;">Import test report</h3>
      <div style="white-space:pre-wrap; line-height:1.5;">
<?php
  echo "Nr of records to import: " . (int)$testReport['records'] . "\n\n";

  echo "Fields that are missing from the import file (optional fields not considered; private not considered):\n";
  $miss = $testReport['missing_fields'] ?? [];
  if (!$miss) echo "  (none)\n\n";
  else { foreach ($miss as $f) echo "  - {$f}\n"; echo "\n"; }

  echo "Fields that do not exist in the application:\n";
  $unk = $testReport['unknown_fields'] ?? [];
  if (!$unk) echo "  (none)\n\n";
  else { foreach ($unk as $f) echo "  - {$f}\n"; echo "\n"; }

  echo "Language field not seem correct (valid: xx OR xx-yy, lowercase):\n";
  $bad = $testReport['lang_invalid'] ?? [];
  if (!$bad) echo "  (none)\n\n";
  else { foreach ($bad as $val => $cnt) echo "  - {$val} ({$cnt})\n"; echo "\n"; }

  echo "Potential duplicates (existing DB record vs CSV line):\n";
  $dups = $testReport['duplicates'] ?? [];
  if (!$dups) {
    echo "  (none)\n";
  } else {
    foreach ($dups as $d) {
      $db = "{$d['db_id']}-{$d['db_company_name']}-{$d['db_name']}";
      $csv = "{$d['csv_line']}-{$d['csv_company_name']}-{$d['csv_name']}";
      echo "  DB: {$db}\n  CSV: {$csv}\n\n";
    }
  }
?>
      </div>
    </div>
  <?php endif; ?>

  <details>
    <summary>Expected columns (any subset accepted)</summary>
    <pre>country, language, company_name, sector, website, name, mf, role, linkedIn/linkedin, general_email, hq_city, email, grupo, (optional: comment, history, status, wrong, last_date, last_template), private (admin only)</pre>
    <p class="note" style="margin-top:8px;">
      Language rules (test): <code>en</code> or <code>en-us</code> (lowercase).
    </p>
  </details>
</section>
<?php include __DIR__ . '/footer.php'; ?>
